/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.threads;

import java.util.concurrent.TimeUnit;

/**
 * @author Antti Villberg
 */
public class Throttler implements Runnable {

	final private IThreadWorkQueue thread;
	final private int interval;

	private boolean isScheduling = false;

	private long[] dispatchTimes;
	private long lastQueueTime;

	private int dispatchTimePointer = 0;

	private Runnable todo;

	public Throttler(IThreadWorkQueue thread, int interval, int queueLength) {

		this.thread = thread;
		this.interval = interval;

		long currentTime = System.currentTimeMillis();

		dispatchTimes = new long[queueLength];
		for(int i=0;i<dispatchTimes.length;i++)
			dispatchTimes[i] = currentTime - interval;

		lastQueueTime = Long.MAX_VALUE;

	}

	public synchronized void run() {

		if(todo != null) {
			thread.asyncExec(todo);
			todo = null;
			ThreadUtils.getTimer().schedule(this, interval, TimeUnit.MILLISECONDS);
		} else {
			isScheduling = false;
		}

	}
	
	public synchronized void schedule(Runnable r) {

		assert(thread.currentThreadAccess());
		
		dispatch(r);

		if(getQueueTime() < interval) {

			isScheduling = true;
			ThreadUtils.getTimer().schedule(this, interval, TimeUnit.MILLISECONDS);

		}

	}

	private long getQueueTime() {
		return lastQueueTime;
	}

	private void dispatch(Runnable runnable) {

		long time = System.currentTimeMillis();
		lastQueueTime = time - dispatchTimes[dispatchTimePointer];
		dispatchTimes[dispatchTimePointer] = time;
		dispatchTimePointer = ((dispatchTimePointer+1) % dispatchTimes.length);

		if(!isScheduling) {
			todo = null;
			runnable.run();
		} else {
			todo = runnable;
		}

	}

}
