/*******************************************************************************
 * Copyright (c) 2007, 2015 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - added getHintsUnsafe
 *******************************************************************************/
/*
 *
 * @author Toni Kalajainen
 */
package org.simantics.utils.datastructures.hints;

import java.util.Map;

import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.threads.IThreadWorkQueue;


public interface IHintObservable {

    /**
     * Check if this observable contains a value for the specified hint key.
     * 
     * @param key the key to check for a value 
     * @return <code>true</code> if there is a value, <code>false</code> otherwise
     */
    boolean containsHint(Key key);

	/**
	 * Get hint
     * @param <E> type of the object bound to the specified key
	 * @param key key
	 * @return the hint or <code>null</code> if does not exist
	 */
	<E> E getHint(Key key);
	
	/**
	 * Get all hints in this context
	 * @return a snapshot of all the hints in this context
	 */
	Map<Key, Object> getHints();
	
	/**
	 * Get all hints in this context without creating defensive copies in the
	 * implementation if possible.
	 * <p>
	 * Use this only when reading and you know what you're doing.
	 * 
	 * @return the backing map of hints in this context without extra copying if possible
	 */
	Map<Key, Object> getHintsUnsafe();
	
	/**
	 * Get all hints whose key is a subclass of <code>clazz</code>
	 * @param <E> 
	 * @param clazz the class
	 * @return key object map
	 */
	<E extends Key> Map<E, Object> getHintsOfClass(Class<E> clazz);
	
	/**
	 * Adds hint listener, which gets events for all hint changes
	 * 
	 * @param listener
	 */
	void addHintListener(IHintListener listener);
	
	/**
	 * Removes hint listener
	 * 
	 * @param listener
	 */
	void removeHintListener(IHintListener listener);
	
	/**
	 * Adds hint listener for a specific key
	 * @param key
	 * @param listener
	 */
	void addKeyHintListener(Key key, IHintListener listener);
	
	/**
	 * Removes hint listener
	 * @param key
	 * @param listener
	 */
	void removeKeyHintListener(Key key, IHintListener listener);
	
	/**
	 * Adds hint listener, which gets events for all hint changes
	 * 
     * @param threadAccess 
	 * @param listener
	 */
	void addHintListener(IThreadWorkQueue threadAccess, IHintListener listener);
	
	/**
	 * Removes hint listener
	 * 
     * @param threadAccess 
	 * @param listener
	 */
	void removeHintListener(IThreadWorkQueue threadAccess, IHintListener listener);
	
	/**
	 * Adds hint listener for a specific key
	 * @param threadAccess 
	 * @param key
	 * @param listener
	 */
	void addKeyHintListener(IThreadWorkQueue threadAccess, Key key, IHintListener listener);
	
	/**
	 * Removes hint listener
     * @param threadAccess 
	 * @param key
	 * @param listener
	 */
	void removeKeyHintListener(IThreadWorkQueue threadAccess, Key key, IHintListener listener);	
	
}
