package org.simantics.tests.modelled.utils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingUtils;
import org.simantics.scl.compiler.commands.CommandSession;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.coverage.CoverageUtils;
import org.simantics.scl.compiler.runtime.RuntimeModule;
import org.simantics.scl.runtime.tuple.Tuple0;
import org.simantics.tests.modelled.ontology.TestsResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class STSSuiteTestCollector {

    private static final Logger LOGGER = LoggerFactory.getLogger(STSSuiteTestCollector.class);
    
    /**
     * TODO: The idea of this class was to collect all the tests from shared libraries and construct
     * JUnit tests out of them programmatically and then run them with JUnit to get results
     * @throws DatabaseException 
     * 
     */
    public static Collection<ModelledSTSSuite> collectTests() throws DatabaseException {
        Collection<ModelledSTSSuite> suitess = Simantics.getSession().syncRequest(new UniqueRead<Collection<ModelledSTSSuite>>() {

            @Override
            public Collection<ModelledSTSSuite> perform(ReadGraph graph) throws DatabaseException {
                
                List<Resource> sharedOntologies = Simantics.applySCL("Simantics/SharedOntologies", "getSharedOntologies", graph, Tuple0.INSTANCE);
                if (LOGGER.isInfoEnabled())
                    LOGGER.info("Found {} shared ontologies from graph",  sharedOntologies.size());
                Collection<ModelledSTSSuite> suites = new HashSet<>();
                TestsResource TESTS = TestsResource.getInstance(graph);
                Layer0 L0 = Layer0.getInstance(graph);
                for (Resource sharedOntology : sharedOntologies) {
                    if (LOGGER.isInfoEnabled())
                        LOGGER.info("Searching {} for modelled tests", graph.getURI(sharedOntology));
                    List<Resource> stsSuites = ModelingUtils.searchByType(graph, sharedOntology, TESTS.STSSuite);
                    for (Resource stsSuite : stsSuites) {
                        try {
                            Collection<Resource> tests = graph.syncRequest(new ObjectsWithType(stsSuite, L0.ConsistsOf, TESTS.STSTest));
                            if (tests.isEmpty())
                                continue;

                            List<ModelledSTSTest> testRunners = new ArrayList<>(tests.size());
                            for (Resource test : tests)
                                testRunners.add(toModelledTest(graph, test));

                            suites.add(toModelledSuite(graph, stsSuite, testRunners));
                        } catch (Exception e) {
                            LOGGER.error("", e);
                        }
                    }
                }
                return suites;
            }
        });
        return suitess;
    }
    

    public static ModelledSTSTest toModelledTest(ReadGraph graph, Resource test) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        TestsResource TESTS = TestsResource.getInstance(graph);
        String testName = graph.getRelatedValue(test, L0.HasName, Bindings.STRING);
        String code = graph.getRelatedValue(test, TESTS.STSTest_definition, Bindings.STRING);
        Integer priority = graph.getPossibleRelatedValue(test, TESTS.STSTest_executionPriority, Bindings.INTEGER);
        Boolean ignored = graph.getPossibleRelatedValue(test, TESTS.ignore, Bindings.BOOLEAN);
        
        String dependencies = graph.getPossibleRelatedValue(test, TESTS.dependencies, Bindings.STRING);
        String[] actualDeps = dependencies.isEmpty() ? new String[0] : dependencies.split(",");
        
        // collect variables
        Collection<Resource> stsVariables = graph.getObjects(test, L0.ConsistsOf);
        Map<String, String> variables = new HashMap<>(stsVariables.size());
        for (Resource stsVariable : stsVariables) {
            String name = graph.getRelatedValue(stsVariable, L0.HasName, Bindings.STRING);
            String value = graph.getRelatedValue(stsVariable, TESTS.STSVariable_definition);
            variables.put(name, value);
        }
        Resource parent = graph.getSingleObject(test, L0.PartOf);
        String parentName;
        String possibleURI = graph.getPossibleURI(parent);
        if (possibleURI != null)
            parentName = possibleURI;
        else
            parentName = graph.getRelatedValue2(parent, L0.HasName, Bindings.STRING);
        return new ModelledSTSTest(testName, parentName, code, priority != null ? priority : -1, ignored != null ? ignored : false, new HashSet<>(Arrays.asList(actualDeps)), variables);
    }

    public static ModelledSTSSuite toModelledSuite(ReadGraph graph, Resource suite, List<ModelledSTSTest> children) throws DatabaseException {
        TestsResource TESTS = TestsResource.getInstance(graph);
        String suiteName = graph.getURI(suite);
        String moduleNameFilter = graph.getPossibleRelatedValue2(suite, TESTS.STSSuite_moduleNameFilter, Bindings.STRING);
        Integer priority = graph.getPossibleRelatedValue2(suite, TESTS.STSTest_executionPriority, Bindings.INTEGER);
        
        Layer0 L0 = Layer0.getInstance(graph);
        Collection<Resource> stsVariables = graph.sync(new ObjectsWithType(suite, L0.ConsistsOf, TESTS.STSVariable));
        Map<String, String> variables = new HashMap<>(stsVariables.size());
        for (Resource stsVariable : stsVariables) {
            String name = graph.getRelatedValue(stsVariable, L0.HasName, Bindings.STRING);
            String value = graph.getRelatedValue(stsVariable, TESTS.STSVariable_definition);
            variables.put(name, value);
        }
        return new ModelledSTSSuite(suiteName, children, moduleNameFilter, priority != null ? priority : -1, variables);
    }

    public static void setTestCoverage(ModelledSTSTest test, CommandSession session) {
        Collection<RuntimeModule> runtimeModules = session.getRuntimeEnvironment().getRuntimeModules();
        List<Module> modules = new ArrayList<>(runtimeModules.size());
        for (RuntimeModule module : runtimeModules)
            modules.add(module.getModule());
        test.setCoverage(CoverageUtils.getCoverage(modules));
        CoverageUtils.resetCoverage(modules);
    }

    public static void setSuiteCoverage(ModelledSTSTest test, ModelledSTSSuite suite, CommandSession session) {
        Collection<RuntimeModule> runtimeModules = session.getRuntimeEnvironment().getRuntimeModules();
        List<Module> modules = new ArrayList<>(runtimeModules.size());
        for (RuntimeModule module : runtimeModules) {
            for (Pattern p : suite.getModuleNameFilterPatterns()) {
                if (p.matcher(module.getModule().getName().toLowerCase()).find()) {
                    modules.add(module.getModule());
                }
            }
        }
        test.setCoverage(CoverageUtils.getCoverage(modules));
        suite.addCoverage(modules);
        CoverageUtils.resetCoverage(modules);
    }
}
