package org.simantics.tests.modelled.junit.v2;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.runner.Description;
import org.junit.runner.notification.Failure;
import org.junit.runner.notification.RunNotifier;
import org.junit.runners.ParentRunner;
import org.junit.runners.model.InitializationError;
import org.simantics.scl.compiler.module.coverage.CombinedCoverage;
import org.simantics.tests.modelled.utils.ModelledSTSSuite;
import org.simantics.tests.modelled.utils.ModelledSTSTest;
import org.simantics.tests.modelled.utils.ModelledSTSTest.CommandSessionVariable;

public class ModelledSTSSuiteRunner extends ParentRunner<ModelledSTSTestRunner> {

    private final ModelledSTSSuite suite;
    private final List<ModelledSTSTestRunner> children;
    private Map<String, List<CommandSessionVariable>> storedVariables = new HashMap<>();

    public ModelledSTSSuiteRunner(ModelledSTSSuite suite) throws InitializationError {
        super(ModelledSTSSuiteRunner.class);
        this.suite = suite;
        this.children = new ArrayList<>(suite.getChildren().size());
        
        // Do possible filtering
        // Filter exclusions
        String exclusionFilter = System.getProperty(ModelledSTSRunner.EXCLUSION_FILTER);
        // Filter inclusions
        String inclusionFilter = System.getProperty(ModelledSTSRunner.INCLUSION_FILTER);
        for (ModelledSTSTest test : suite.getSortedChildren()) {
            boolean add = true;
            if (exclusionFilter != null) {
                String[] filters = exclusionFilter.split(",");
                if (startsWithAny(test, filters)) {
                    add = false;
                }
            }
            if (inclusionFilter != null) {
                String[] filters = inclusionFilter.split(",");
                if (!startsWithAny(test, filters)) {
                    add = false;
                }
            }
            if (add)
                children.add(new ModelledSTSTestRunner(test));
        }
    }

    private static boolean startsWithAny(ModelledSTSTest test, String[] filters) {
        for (String filter : filters) {
            if (test.getName().contains(filter)) {
                return true;
            }
        }
        return false;
    }
    
    @Override
    protected String getName() {
        return suite.getName();
    }

    @Override
    protected List<ModelledSTSTestRunner> getChildren() {
        return children;
    }

    @Override
    protected Description describeChild(ModelledSTSTestRunner child) {
        return child.getDescription();
    }

    @Override
    protected void runChild(ModelledSTSTestRunner child, RunNotifier notifier) {
        Description description = describeChild(child);
        if (isIgnored(child)) {
            notifier.fireTestIgnored(description);
        } else {
            try {
                List<CommandSessionVariable> variables = new ArrayList<>();
                for (String dep : child.getTest().getDependencies()) {
                    List<CommandSessionVariable> storedVars = storedVariables.get(dep);
                    if (storedVars != null) {
                        variables.addAll(storedVars);
                    }
                }
                notifier.fireTestStarted(description);
                List<CommandSessionVariable> newVars = child.runWithVars(variables);
//                TODO: FIX THIS BY NOT ADDING ALL VARIABLES TO MEMORY (CAN BE HUGE)
//                storedVariables.put(child.getTest().getName(), newVars);
                notifier.fireTestFinished(description);
            } catch (Throwable e) {
                notifier.fireTestFailure(new Failure(description, e));
            }
        } 
    }

    @Override
    protected boolean isIgnored(ModelledSTSTestRunner child) {
        return child.isIgnored();
    }

    public CombinedCoverage getCoverage() {
        return suite.getCoverage();
    }

}
