/*******************************************************************************
 * Copyright (c) 2018 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.structural2.variables;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.utils.datastructures.Pair;
import org.slf4j.LoggerFactory;

import gnu.trove.set.hash.THashSet;

/**
 * @author Antti Villberg
 * @since 1.36.0
 */
public class FixedConnection implements Connection {

    private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(FixedConnection.class);

    /*
     * This is the parent of the component to be connected
     */
    private final Variable procedural;

    private final Collection<Pair<String, Resource>> cps = new ArrayList<Pair<String, Resource>>();

    public FixedConnection(Variable procedural) {
        this.procedural = procedural;
    }

    public void addAll(List<Pair<String, Resource>> cps) throws DatabaseException {
        /*
         * For interface connections the name is null
         */
        for (Pair<String, Resource> cp : cps) {
            this.cps.add(cp);
        }
    }

    public int size() {
        return cps.size();
    }

    public void addConnectionDescriptors(ReadGraph graph, Variable curConfiguration,
            Collection<VariableConnectionPointDescriptor> result) throws DatabaseException {
        for (Pair<String, Resource> cpzz : cps) {
            // This is a connection to an interface terminal. It is handled by
            // ConnectionBrowser in separate logic. We should never have gotten this far
            if (cpzz.first == null) {
                String message = "Lifted connection was not resolved. Child = " + procedural.getURI(graph);
                throw new DatabaseException(message);
            }
            result.add(new PairConnectionDescriptor(curConfiguration, cpzz));
        }
    }

    @Override
    public Collection<Variable> getConnectionPoints(ReadGraph graph, Resource relationType) throws DatabaseException {
        Set<Variable> result = new THashSet<Variable>();
        for (Pair<String, Resource> cp : cps) {
            Variable component = cp.first == null ? procedural : procedural.getChild(graph, cp.first);
            Variable cp2 = component.getPossibleProperty(graph, cp.second);
            if (cp2 != null)
                for (VariableConnectionPointDescriptor desc : ConnectionBrowser.flatten(graph, component, cp.second,
                        relationType)) {
                    result.add(desc.getVariable(graph));
                }
            else
                LOGGER.warn("no cp " + cp.first + " for " + component.getURI(graph));
        }
        return result;
    }

    @Override
    public Collection<String> getConnectionPointURIs(ReadGraph graph, Resource relationType) throws DatabaseException {
        Set<String> result = new THashSet<String>();
        for (Pair<String, Resource> cp : cps) {
            Variable component = cp.first == null ? procedural : procedural.getChild(graph, cp.first);
            Variable cp2 = component.getPossibleProperty(graph, cp.second);
            if (cp2 != null) {
                for (VariableConnectionPointDescriptor desc : ConnectionBrowser.flatten(graph, component, cp.second,
                        relationType)) {
                    result.add(desc.getURI(graph));
                }
            } else {
                logWarn(graph, cp, component, procedural);
            }
        }
        return result;
    }

    @Override
    public Collection<VariableConnectionPointDescriptor> getConnectionPointDescriptors(ReadGraph graph,
            Resource relationType) throws DatabaseException {
        Set<VariableConnectionPointDescriptor> result = new THashSet<>();
        for (Pair<String, Resource> cp : cps) {
            Variable component = cp.first == null ? procedural : procedural.getChild(graph, cp.first);
            Variable cp2 = component.getPossibleProperty(graph, cp.second);
            if (cp2 != null) {
                result.addAll(ConnectionBrowser.flatten(graph, component, cp.second, relationType));
            } else {
                logWarn(graph, cp, component, procedural);
            }
        }
        return result;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((cps == null) ? 0 : cps.hashCode());
        result = prime * result + ((procedural == null) ? 0 : procedural.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        FixedConnection other = (FixedConnection) obj;
        if (cps == null) {
            if (other.cps != null)
                return false;
        } else if (!cps.equals(other.cps))
            return false;
        if (procedural == null) {
            if (other.procedural != null)
                return false;
        } else if (!procedural.equals(other.procedural))
            return false;
        return true;
    }

    private static void logWarn(ReadGraph graph, Pair<String, Resource> cp, Variable base, Variable procedural) throws DatabaseException {
        LOGGER.warn("no cp " + cp.first + " for " + base.getURI(graph));
        LOGGER.warn("    proc: " + procedural.getURI(graph));
        LOGGER.warn("    rel: " + graph.getURI(cp.second));
        LOGGER.warn("    base: " + base.getURI(graph));
    }

}
