/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.structural2.validate;

import gnu.trove.set.hash.THashSet;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.layer0.Layer0;
import org.simantics.structural.stubs.StructuralResource2;

public class ValidateStructural {

	ReadGraph g;
	Layer0 b;
	StructuralResource2 sr;
	boolean validateTypesRecursively;
	THashSet<Resource> validatedTypes = new THashSet<Resource>();
	
	IErrorReporter reporter = new StdoutReporter();
	
	public ValidateStructural(ReadGraph g, boolean validateTypesRecursively) {
		this.g = g;
		this.b = Layer0.getInstance(g);
		this.sr = StructuralResource2.getInstance(g);
		this.validateTypesRecursively = validateTypesRecursively;
	}
	
	public void setReporter(IErrorReporter reporter) {
		this.reporter = reporter;
	}
	
	/**
	 * Checks that all components reachable from rootComposite recursively by ConsistsOf
	 * have unique names.
	 * @param rootComposite
	 * @throws DatabaseException
	 */
	private void checkUniqueNames(Resource rootComposite) throws DatabaseException {
		THashSet<String> names = new THashSet<String>();
		checkUniqueNames(rootComposite, names);
	}

	private void checkUniqueNames(Resource composite, THashSet<String> names) throws DatabaseException {
		for(Resource component : g.getObjects(composite, b.ConsistsOf)) {
			String name = null;
			try {
				name = g.getRelatedValue(component, b.HasName);
			} catch(DatabaseException e) {
				reporter.report("Component does not have a name or there are two names.", component);
			}
			if(name != null)
				if(!names.add(name))
					reporter.report("Name '" + name + "' is not unique.", component);
			if(g.isInstanceOf(component, sr.Composite))
				checkUniqueNames(composite, names);
		}
	}
	
	public void validateComponent(Resource component) throws DatabaseException {
		if(!g.isInstanceOf(component, sr.Component)) {
			reporter.report(NameUtils.getSafeName(g, component) +
					" is not a component.", component);
			return;
		}
		if(g.isInstanceOf(component, sr.Composite))
			for(Resource child : g.getObjects(component, b.ConsistsOf))
				validateComponent(child);
		else
			; // TODO
	}
	
	public void validateRootComposite(Resource rootComposite) throws DatabaseException {
		checkUniqueNames(rootComposite);
		validateComponent(rootComposite);
	}
	
	public void validateComponentType(Resource componentType) throws DatabaseException {
		Resource rootComposite = null;
		try {
			rootComposite = g.getPossibleObject(componentType, sr.IsDefinedBy);
		} catch(ManyObjectsForFunctionalRelationException e) {
			reporter.report("A user component " + NameUtils.getSafeName(g, componentType) 
					+ " has multiple definitions.", componentType);
		}
		if(rootComposite != null)
			validateRootComposite(rootComposite);	
	}
	
}
