package org.simantics.spreadsheet.graph;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.simantics.spreadsheet.Range;
import org.simantics.spreadsheet.graph.parser.ast.AstRange;

@SuppressWarnings("rawtypes")
public class SpreadsheetEngine implements SpreadsheetElement, SheetNode {

	private static final long serialVersionUID = -5246063647558595642L;
	
	private static final String LINES = "Lines";
	
	private final SpreadsheetBook book;
	private final String name;
	private final int id;

	public SpreadsheetLines lines;
	
	transient public Map<String,Object> rangeCache;

	public Map<String,Object> getRangeCache() {
		if(rangeCache == null) rangeCache = new HashMap<String,Object>();
		return rangeCache;
	}
	
	public Object getCachedRange(AstRange range) {
		if(range.sheetName != null) return null;
		return getRangeCache().get(range.first + ":" + range.second); 
	}
	
	public void cacheRange(AstRange range, Object value) {
		if(range.sheetName != null) return;
		getRangeCache().put(range.first + ":" + range.second, value);
	}

	public SpreadsheetEngine(SpreadsheetBook book, String name) {
		this.book = book;
		this.name = name;
		this.id = book.getNewId(this);
		this.lines = new SpreadsheetLines(this, LINES);
	}
	
	public SpreadsheetBook getBook() {
		return book;
	}
	
	public int getId() {
		return id;
	}
	
	Object resolve(String[] parts, int index) {
		
		String part = parts[index];
		if(!part.equals(LINES)) return null;
		
		if(index == parts.length-1) return lines;
		
		return lines.resolve(parts, index+1);

	}
	
	@Override
	public String getName() {
		return name;
	}
	
	@Override
	public Map<String, SheetNode> getChildren() {
		return Collections.singletonMap(LINES, lines);
	}
	
	@Override
	public Map<String, SheetNode> getProperties() {
		return Collections.emptyMap();
	} 
	
	public Object ensureSubprocess(String[] path, int index) {
		
		String name = path[index];
		if(!LINES.equals(name)) throw new IllegalStateException();
		if(index == path.length - 1) return lines;
		return lines.ensureSubprocess(path, index+1);
		
	}
	
	public SpreadsheetLine getLine(int row) {
		assert(lines.nodes.size() == 1);
		SpreadsheetLines root = lines.nodes.values().iterator().next();
		return root.getLine(row);
	}

	@Override
	public void accept(SpreadsheetVisitor v) {
		v.visit(this);
	}
	
	public Range actualRange(Range r) {
		if(r.isFullRows()) {
			SpreadsheetLines root = lines.nodes.values().iterator().next();
			Range result = new Range(r);
			result.startRow = 0;
			result.endRow = root.getMaxRow(); 
			return result;
		} else {
			return r;
		}
	}

    @Override
    public Optional<SpreadsheetElement> getParent() {
        return Optional.of(book);
    }

    @Override
    public List<SpreadsheetElement> getSpreadsheetChildren() {
        return Collections.singletonList(lines);
    }

    @Override
    public void remove(SpreadsheetElement child) {
        
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((book == null) ? 0 : book.hashCode());
        result = prime * result + ((name == null) ? 0 : name.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        SpreadsheetEngine other = (SpreadsheetEngine) obj;
        if (book == null) {
            if (other.book != null)
                return false;
        } else if (!book.equals(other.book))
            return false;
        if (name == null) {
            if (other.name != null)
                return false;
        } else if (!name.equals(other.name))
            return false;
        return true;
    }
	
	
}