package org.simantics.simulator.toolkit;

import org.simantics.simulator.ExperimentState;

/**
 * @author Antti Villberg
 * @since 1.34.0
 */
public class StandardExperimentStates implements ExperimentState {

    /**
     * The experiment context has been created.
     *
     * <p>
     * Allowed successor states: {@link Instantiated}, {@link ToBeDisposed}
     */
    public interface Created extends ExperimentState {}

    /**
     * The experiment context has been instantiated, i.e. underlying experimentation
     * resources have been acquired. Some experiment setup (e.g. setting certain
     * values) can be done in this state that cannot be touched once the experiment
     * is initialized.
     *
     * <p>
     * Allowed successor states: {@link Initializing}, {@link ToBeDisposed}
     */
    public interface Instantiated extends ExperimentState {}

    /**
     * The experiment context is in the process of being initialized. This means
     * that any initial conditions or related data is being loaded into the
     * simulator/solver/what ever system that runs the experiment.
     * 
     * If the initialization fails due to an irrecoverable system error, the
     * experiment shall move to {@link Failure} state. If the initialization fails
     * due to a recoverable error in user input, the experiment shall move to
     * {@link Instantiated} state.
     *
     * <p>
     * Allowed successor states: {@link Initialized}, {@link Instantiated},
     * {@link Failure}
     */
    public interface Initializing extends ExperimentState {}

    /**
     * The experiment context has been initialized, i.e. the underlying
     * experimentation resources have been both acquired and initialized with a
     * specific state.
     * 
     * <p>
     * Allowed successor states: {@link Stopped}, {@link Running},
     * {@link ToBeDisposed}
     */
    public interface Initialized extends ExperimentState {}

    /**
     * The experiment shall be ran until it reaches an objective such as
     * steady-state or running for a duration or until the state changes to
     * {@link Stopped}.
     *
     * <p>
     * Allowed successor states: {@link Stopped}, {@link ToBeDisposed}
     */
    public interface Running extends ExperimentState {}

    /**
     * The experiment shall remain stopped. Everything in the experiment context
     * should remain constant while in this state.
     *
     * <p>
     * Allowed successor states: {@link Running}, {@link ToBeDisposed}
     */
    public interface Stopped extends ExperimentState {}

    /**
     * Moving into this state marks the beginning of the disposal of this
     * experiment. The imminent disposal is irreversible and cannot be vetoed
     * or interrupted.
     *
     * <p>
     * Allowed successor states: {@link Disposing}
     */
    public interface ToBeDisposed extends ExperimentState {}

    /**
     * The experiment context is being disposed, i.e. underlying experimentation
     * resources are being freed and disposed of accordingly.
     *
     * <p>
     * Allowed successor states: {@link Disposing}
     */
    public interface Disposing extends ExperimentState {}

    /**
     * The experiment has been completely disposed and cannot any longer be used for
     * anything.
     *
     * <p>
     * Allowed successor states: none, this is a final state
     */
    public interface Disposed extends ExperimentState {}

    /**
     * The experiment implementation has ran into a fatal failure. The experiment
     * context can still be accessed at this point but the experiment can only move
     * into {@link Disposed} state.
     *
     * <p>
     * Allowed successor states: {@link Disposed}
     */
    public interface Failure extends ExperimentState {}

    public static final ExperimentState CREATED = new Created() {};
    public static final ExperimentState INSTANTIATED = new Instantiated() {};
    public static final ExperimentState INITIALIZING = new Initializing() {};
    public static final ExperimentState INITIALIZED = new Initialized() {};
    public static final ExperimentState RUNNING = new Running() {};
    public static final ExperimentState STOPPED = new Stopped() {};
    public static final ExperimentState TO_BE_DISPOSED = new ToBeDisposed() {};
    public static final ExperimentState DISPOSING = new Disposing() {};
    public static final ExperimentState DISPOSED = new Disposed() {};

}