/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.structural.schema;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.objmap.backward.IBackwardMapping;
import org.simantics.objmap.bidirectional.IBidirectionalMappingRule;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.forward.IForwardMapping;
import org.simantics.objmap.graph.schema.ILinkType;
import org.simantics.objmap.structural.IStructuralObject;
import org.simantics.objmap.structural.StructuralResource;



public class SimpleLinkType implements ILinkType<StructuralResource,IStructuralObject> {
    
    static Logger LOGGER = LoggerFactory.getLogger(SimpleLinkType.class);
    
    public Resource domainType;
    public Class<?> rangeType;
    ArrayList<IBidirectionalMappingRule<StructuralResource,IStructuralObject>> rules;
    
    public SimpleLinkType(Resource domainType, Class<?> rangeType,
            ArrayList<IBidirectionalMappingRule<StructuralResource,IStructuralObject>> rules) {
        this.domainType = domainType;
        this.rangeType = rangeType;
        this.rules = rules;
    }

    public SimpleLinkType(Resource domainType, Class<?> rangeType) {
        this(domainType, rangeType, new ArrayList<IBidirectionalMappingRule<StructuralResource,IStructuralObject>>());
    }

    /**
     * Adds a new rule to this link type that is enforced
     * during updates.
     */
    public void addRule(IBidirectionalMappingRule<StructuralResource,IStructuralObject> rule) {
        rules.add(rule);
    }
    
    @Override
    public StructuralResource createDomainElement(WriteGraph g, IStructuralObject rangeElement)
            throws MappingException {
        try {
            if(LOGGER.isTraceEnabled())
                LOGGER.trace("SimpleLinkType.createDomainElement " +
                        rangeElement.toString()
                );
            if (rangeElement.getContext().size() == 0) {
            	// there is no context, this not a structural resource / object.
            	Resource result = newResource(g, domainType);
            	return new StructuralResource(g,result);
            } else {
            	if (rangeElement.getContext().size() == 1 && rangeElement.getContext().get(0).equals(rangeElement)) {
            		// Structural object's context is itself, we are instantiating a new structural model.
            		Resource type = rangeElement.getType();
            		Resource result = newResource(g, type);
                	return new StructuralResource(g,result,result);
            	} else {
            		// Structural object's context is not itself, which means that the object is inside of a structural model.
            		// At the moment we do not support modifying instantiated structural models.
            		throw new MappingException("Cannot create a new StucturalObject " + rangeElement + " " + rangeElement.getClass());
            	}
            }
        } catch(DatabaseException e) {
            throw new MappingException(e);
        }
    }
    
    protected Resource newResource(WriteGraph g, Resource type) throws DatabaseException {
    	Resource result = g.newResource();
    	g.claim(result, Layer0.getInstance(g).InstanceOf, null, type);
    	return result;
    }
    @Override
    public IStructuralObject createRangeElement(ReadGraph g, StructuralResource domainElement)
            throws MappingException {
        try {
            if(LOGGER.isTraceEnabled())
                try { 
                    LOGGER.trace("SimpleLinkType.createRangeElement " + NameUtils.getSafeName(g, domainElement.getResource()));
                } catch(DatabaseException e) {
                    throw new MappingException(e);
                }
            IStructuralObject result = (IStructuralObject)rangeType.newInstance();
            if (domainElement.getContext().size() == 1) {
            	if (domainElement.getContext().get(0).equals(domainElement.getResource()))
            		result.setContext(Collections.singletonList(result));
            	else {
            		//result.setContext(result); 
            	}
            }
            return result;
        } catch (InstantiationException e) {
            throw new MappingException(e);
        } catch (IllegalAccessException e) {
            throw new MappingException(e);
        }
    }
    
    @SuppressWarnings("unchecked")
	public void createDomain(WriteGraph graph, IBackwardMapping<StructuralResource,IStructuralObject> mapping, StructuralResource domainElement, IStructuralObject rangeElement) throws MappingException {
    	if (domainElement.isStructuralRoot())
    		// FIXME: this is nasty, but when a structural model is instantiated by creating new IStructuralObject, its related objects must be read from the graph first, or otherwise the objects would be deleted from the graph.
    		//        as a side effect, if the new IStructuralObject has any properties set, those properties are set to null (because the graph does not contain those values).
    		// 
    		updateRange(graph, (IForwardMapping<StructuralResource, IStructuralObject>)mapping, domainElement, rangeElement);
    	updateDomain(graph, mapping, domainElement, rangeElement);
    };
    
    
    @Override
    public void createRange(ReadGraph graph, IForwardMapping<StructuralResource, IStructuralObject> mapping, StructuralResource domainElement, IStructuralObject rangeElement) throws MappingException {
    	if (rangeElement.getContext().size() == 0 && domainElement.getContext().size() > 0) {
    		List<IStructuralObject> ctx = new ArrayList<IStructuralObject>(domainElement.getContext().size());
			try {
				List<Resource> context = new ArrayList<Resource>();
				for (int i = 0; i <domainElement.getContext().size(); i++) {
					context.add(domainElement.getContext().get(i));
					IStructuralObject ctxObj = mapping.get(new StructuralResource(graph,context.get(context.size()-1),context));
					if (ctxObj == null) throw new MappingException("Cannot resolve range context for domain element " + domainElement);
					ctx.add(ctxObj);
				}
				//ctx = mapping.get(new StructuralResource(graph,context.get(context.size()-1),context));
			} catch (DatabaseException e) {
				throw new MappingException(e);
			}
    		if (ctx.size() == 0)
    			throw new MappingException("Cannot find context for structural object, " + domainElement);
    		rangeElement.setContext(ctx);
    	}
    	updateRange(graph, mapping, domainElement, rangeElement);
    }
    
    public boolean updateDomain(WriteGraph g, IBackwardMapping<StructuralResource,IStructuralObject> map, StructuralResource domainElement, IStructuralObject rangeElement) throws MappingException {
        if(LOGGER.isTraceEnabled())
            try { 
                LOGGER.trace("SimpleLinkType.updateDomain " +
                        NameUtils.getSafeName(g, domainElement.getResource()) + " " +
                        rangeElement.toString()
                        );
            } catch(DatabaseException e) {
                throw new MappingException(e);
            }
        
        boolean updated = false;
        for(IBidirectionalMappingRule<StructuralResource, IStructuralObject> rule : rules)
        	updated |= rule.updateDomain(g, map, domainElement, rangeElement);
        return updated;
    }
    
    public boolean updateRange(ReadGraph g, IForwardMapping<StructuralResource, IStructuralObject> map, StructuralResource domainElement, IStructuralObject rangeElement) throws MappingException {
    
        if(LOGGER.isTraceEnabled())
            try { 
                LOGGER.trace("SimpleLinkType.updateRange " +
                		NameUtils.getSafeName(g, domainElement.getResource()) + " " +
                        rangeElement.toString()
                        );
            } catch(DatabaseException e) {
                throw new MappingException(e);
            }
        
        boolean updated = false;
        for(IBidirectionalMappingRule<StructuralResource, IStructuralObject> rule : rules)
            updated |= rule.updateRange(g, map, domainElement, rangeElement);
        return updated;
    }
}
