/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.graph.schema;

import java.util.ArrayList;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.objmap.backward.IBackwardMapping;
import org.simantics.objmap.bidirectional.IBidirectionalMappingRule;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.forward.IForwardMapping;


/**
 * A link type that is associated with single domain and range type (class).
 * SimpleLinkType is composed of simpler rules whose combination determines
 * its update policy.
 * @author Hannu Niemist�
 */
public class SimpleLinkType<Range> implements ILinkType<Resource,Range> {
    
    static Logger LOGGER = LoggerFactory.getLogger(SimpleLinkType.class);
    
    public Resource domainType;
    public Class<?> rangeType;
    ArrayList<IBidirectionalMappingRule<Resource, Range>> rules;
    
    public SimpleLinkType(Resource domainType, Class<?> rangeType,
            ArrayList<IBidirectionalMappingRule<Resource, Range>> rules) {
        this.domainType = domainType;
        this.rangeType = rangeType;
        this.rules = rules;
    }

    public SimpleLinkType(Resource domainType, Class<?> rangeType) {
        this(domainType, rangeType, new ArrayList<IBidirectionalMappingRule<Resource, Range>>());
    }

    /**
     * Adds a new rule to this link type that is enforced
     * during updates.
     */
    public void addRule(IBidirectionalMappingRule<Resource, Range> rule) {
        rules.add(rule);
    }
    
    @Override
    public Resource createDomainElement(WriteGraph g, Range rangeElement)
            throws MappingException {
        try {
            if(LOGGER.isTraceEnabled())
                LOGGER.trace("SimpleLinkType.createDomainElement " +
                        rangeElement.toString()
                );
            Resource result = g.newResource();
            g.claim(result, Layer0.getInstance(g).InstanceOf, null, domainType);
            return result;
        } catch(DatabaseException e) {
            throw new MappingException(e);
        }
    }
    @SuppressWarnings("unchecked")
	@Override
    public Range createRangeElement(ReadGraph g, Resource domainElement)
            throws MappingException {
        try {
            if(LOGGER.isTraceEnabled())
                try { 
                    LOGGER.trace("SimpleLinkType.createRangeElement " +
                    		NameUtils.getSafeName(g, domainElement)
                            );
                } catch(DatabaseException e) {
                    throw new MappingException(e);
                }
            return (Range)rangeType.newInstance();
        } catch (InstantiationException e) {
            throw new MappingException(e);
        } catch (IllegalAccessException e) {
            throw new MappingException(e);
        }
    }
    
    public void createDomain(WriteGraph graph, IBackwardMapping<Resource,Range> mapping, Resource domainElement, Range rangeElement) throws MappingException {
    	updateDomain(graph, mapping, domainElement, rangeElement);
    };
    
    public void createRange(ReadGraph graph, org.simantics.objmap.forward.IForwardMapping<Resource,Range> mapping, Resource domainElement, Range rangeElement) throws MappingException {
    	updateRange(graph, mapping, domainElement, rangeElement);
    };
    
    public boolean updateDomain(WriteGraph g, IBackwardMapping<Resource,Range> map, Resource domainElement, Range rangeElement) throws MappingException {
        if(LOGGER.isTraceEnabled())
            try { 
                LOGGER.trace("SimpleLinkType.updateDomain " +
                        NameUtils.getSafeName(g, domainElement) + " " +
                        rangeElement.toString()
                        );
            } catch(DatabaseException e) {
                throw new MappingException(e);
            }
        
        boolean updated = false;
        for(IBidirectionalMappingRule<Resource, Range> rule : rules)
        	updated |= rule.updateDomain(g, map, domainElement, rangeElement);
        return updated;
    }
    
    public boolean updateRange(ReadGraph g, IForwardMapping<Resource,Range> map, Resource domainElement, Range rangeElement) throws MappingException {
    
        if(LOGGER.isTraceEnabled())
            try { 
                LOGGER.trace("SimpleLinkType.updateRange " +
                		NameUtils.getSafeName(g, domainElement) + " " +
                        (rangeElement.getClass().getName() + "@" + Integer.toHexString(rangeElement.hashCode()))
                        );
            } catch(DatabaseException e) {
                throw new MappingException(e);
            }
        
        boolean updated = false;
        for(IBidirectionalMappingRule<Resource, Range> rule : rules)
            updated |= rule.updateRange(g, map, domainElement, rangeElement);
        return updated;
    }
}
