/*******************************************************************************
 * Copyright (c) 2017 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.scl;

import java.io.StringReader;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.scl.compiler.commands.CommandSession;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.runtime.reporting.SCLReportingHandler;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.ui.workbench.WorkbenchUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 * @since 1.31.0
 */
public class SCLScripts {

	private static final Logger LOGGER = LoggerFactory.getLogger(SCLScripts.class);

	private static final String SCL_SCRIPT_CONSOLE_ID = "org.simantics.scl.ui.scriptConsole";

	/**
	 * @param processor database handle
	 * @param script the script to validate
	 * @return <code>null</code> if ok to run script, otherwise a problem description
	 * @throws DatabaseException
	 */
	public static String canRunScript(RequestProcessor processor, final Resource script) throws DatabaseException {
		return null;
	}

	public static void runScriptWithProgress(String scriptName, String scriptText, CommandSession session, SCLReportingHandler handler) {
		Job job = new Job("Run SCL Script") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				runScriptWithProgress(monitor, scriptName, scriptText, session, handler);
				return Status.OK_STATUS;
			}
		};
		job.setUser(true);
		job.schedule();
	}

	public static void runScriptWithProgress(IProgressMonitor monitor, String scriptName, String scriptText, CommandSession session, SCLReportingHandler handler) {
		monitor.beginTask(scriptName, IProgressMonitor.UNKNOWN);
		try {
			session.execute(new StringReader(scriptText), handler);
		} finally {
			monitor.done();
		}
	}

	public static Pair<CommandSession, SCLReportingHandler> getOrCreateConsoleCommandSession() {
		return getSCLConsoleCommandSession(true);
	}

	public static Pair<CommandSession, SCLReportingHandler> getSCLConsoleCommandSession(boolean createIfNecessary) {
		IWorkbenchPart part;
		SCLReportingHandler handler = SCLReportingHandler.DEFAULT_WITHOUT_ECHO;
		try {
			part = createIfNecessary
					? WorkbenchUtils.showView(SCL_SCRIPT_CONSOLE_ID, IWorkbenchPage.VIEW_VISIBLE)
					: WorkbenchUtils.findView(SCL_SCRIPT_CONSOLE_ID);
			if (part != null)
				return Pair.make(new CommandSession(SCLOsgi.MODULE_REPOSITORY, handler), part.getAdapter(SCLReportingHandler.class));
		} catch (PartInitException e) {
			LOGGER.error("Failed to open SCL Console view. Using new CommandSession, reporting to stdout via Logger.", e);
		}
		return Pair.make(new CommandSession(SCLOsgi.MODULE_REPOSITORY, handler), SCLReportingHandler.DEFAULT);
	}

}
