package org.simantics.modeling.ui.componentTypeEditor;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.ui.IEditorInput;
import org.simantics.NameLabelMode;
import org.simantics.NameLabelUtil;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ui.features.EditorNamingService2;
import org.simantics.ui.workbench.IEditorNamingService2;
import org.simantics.ui.workbench.IResourceEditorInput;

/**
 * Tries to:
 * <ul>
 * <li>resolve input path from model configuration to put in the tooltip</li>
 * <li>add (model-name) suffix to editor tooltip to tell apart editors with same title.</li>
 * </ul>
 * 
 * The goal tooltip format is: path/input-name (model-name)
 * 
 * @author Tuukka Lehtonen
 */
public class ProceduralComponentTypeEditorNamingService extends EditorNamingService2 implements IEditorNamingService2 {

    @Override
    public String getName(ReadGraph graph, String editorId, IEditorInput input) throws DatabaseException {
        if (input instanceof IResourceEditorInput) {
            Resource code = ((IResourceEditorInput) input).getResource();
            if (code != null) {
                return truncated(getPropertyOwnerName(graph, code) + " (Code)", 256);
            }
        }
        return "Unsupported input: " + input;
    }

    @Override
    public String getToolTipText(ReadGraph graph, String editorId, IEditorInput input) throws DatabaseException {
        if (input instanceof IResourceEditorInput) {
            Resource code = ((IResourceEditorInput) input).getResource();
            if (code != null) {
                StringBuilder sb = new StringBuilder();
                getTooltip(graph, editorId, (IResourceEditorInput) input, sb);
                return sb.toString();
            }
        }
        return "Unsupported input: " + input;
    }

    private StringBuilder getTooltip(ReadGraph graph, String editorId, IResourceEditorInput input, StringBuilder sb) throws DatabaseException {
        Resource r = ((IResourceEditorInput) input).getResource();
        NameLabelMode mode = NameLabelUtil.getNameLabelMode(graph);

        String name = getName(graph, editorId, input);
        Layer0 L0 = Layer0.getInstance(graph);
        Resource owner = graph.getPossibleObject(r, L0.PropertyOf);
        if (owner != null) {
            Resource root = graph.syncRequest(new PossibleIndexRoot(owner));
            if (root != null) {
                ResourceArray path = getPathInIndexRoot(graph, owner);
                for (int i = path.size() - 2; i > 0; --i) {
                    String segment = NameLabelUtil.modalName(graph, path.get(i), mode);
                    if (segment.contains("/"))
                        segment = "\"" + segment + "\"";

                    sb.append(segment).append("/");
                }
                sb.append(name);
                String rootLabel = NameLabelUtil.modalName(graph, root, mode);
                sb.append(" (" + rootLabel + ")");
                return sb;
            }
        }
        sb.append(name);
        return sb;
    }

    private String getPropertyOwnerName(ReadGraph graph, Resource property) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        Resource owner = graph.getPossibleObject(property, L0.PropertyOf);
        if (owner == null)
            return "No owner (Code)";
        return graph.getPossibleRelatedValue(owner, L0.HasName);
    }

    private static ResourceArray getPathInIndexRoot(ReadGraph graph, Resource r) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        List<Resource> path = new ArrayList<Resource>();
        while (true) {
            path.add(r);
            if (graph.isInstanceOf(r, L0.IndexRoot))
                return new ResourceArray(path);
            Resource partOf = graph.getPossibleObject(r, L0.PartOf);
            if (partOf == null)
                return ResourceArray.EMPTY;
            r = partOf;
        }
    }

}