/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.componentTypeEditor;

import java.util.function.Supplier;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.editors.text.TextEditor;
import org.eclipse.ui.texteditor.ITextEditorActionConstants;
import org.eclipse.ui.texteditor.StatusLineContributionItem;
import org.simantics.Simantics;
import org.simantics.db.Session;
import org.simantics.db.common.request.ParametrizedRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.combinations.Combinators;
import org.simantics.scl.ui.editor.SharedTextColorsNew;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.ui.workbench.ResourceEditorSupport;
import org.simantics.ui.workbench.TitleUpdater;
import org.simantics.ui.workbench.TitleWithParentNameRequest;
import org.simantics.ui.workbench.ToolTipRequest;
import org.simantics.ui.workbench.editor.input.InputValidationCombinators;


/**
 * @author Antti Villberg;
 */
public class PGraphEditor extends TextEditor {

    private boolean disposed = false;

    private ParametrizedRead<IResourceEditorInput, Boolean> INPUT_VALIDATOR =
            Combinators.compose(
                    InputValidationCombinators.hasURI(),
                    InputValidationCombinators.extractInputResource()
            );

    private ResourceEditorSupport support;

    public PGraphEditor() {
        super();
        preInitialize();
    }

    protected void preInitialize() {
        //showOverviewRuler();
        setDocumentProvider(new PGraphEditorDocumentProvider());
        Display display = PlatformUI.getWorkbench().getDisplay();
        setSourceViewerConfiguration(new PGraphSourceViewerConfigurationNew(
                display, new SharedTextColorsNew(display)));
    }

    protected ParametrizedRead<IResourceEditorInput, Boolean> getInputValidator() {
        return INPUT_VALIDATOR;
    }

    @Override
    public void init(IEditorSite site, IEditorInput input)
            throws PartInitException {
        super.init(site, input);

        support = new ResourceEditorSupport(this, getInputValidator());
        support.activateValidation();
        try {
            getResourceInput().init(null);
        } catch (DatabaseException e) {
            throw new PartInitException("Failed to initialize " + input, e);
        }
    }

    protected IResourceEditorInput getResourceInput() {
        return (IResourceEditorInput) getEditorInput();
    }

    @Override
    public void createPartControl(Composite parent) {
        super.createPartControl(parent);
        StatusLineContributionItem statusLineContribution = new StatusLineContributionItem(
                ITextEditorActionConstants.STATUS_CATEGORY_INPUT_POSITION,
                true, 14);
        setStatusField(statusLineContribution,
                ITextEditorActionConstants.STATUS_CATEGORY_INPUT_POSITION);
        getEditorSite().getActionBars().getStatusLineManager().add(statusLineContribution);
        getEditorSite().getActionBars().updateActionBars();
        updatePartName();
    }

    protected void updatePartName() {
        setPartName(getEditorInput().getName());

        Session session = Simantics.peekSession();
        if (session != null) {
            Supplier<Boolean> disposedCallback = this::isDisposed;
            session.asyncRequest(
                    new TitleWithParentNameRequest(getResourceInput()),
                    new TitleUpdater(getSite().getShell().getDisplay(), this::setPartName, disposedCallback));
            session.asyncRequest(
                    new ToolTipRequest(getSite().getId(), getResourceInput()),
                    new TitleUpdater(getSite().getShell().getDisplay(), this::setTitleToolTip, disposedCallback));
        }
    }

    @Override
    public void dispose() {
        disposed = true;
        if (support != null) {
            support.dispose();
            support = null;
        }
        super.dispose();
    }

    public boolean isDisposed() {
        return disposed;
    }

}