/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.actions.e4;

import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import javax.inject.Named;

import org.eclipse.e4.core.di.annotations.Execute;
import org.eclipse.e4.ui.model.application.ui.basic.MPart;
import org.eclipse.e4.ui.services.IServiceConstants;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.e4.compatibility.CompatibilityEditor;
import org.eclipse.ui.part.MultiPageEditorPart;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.common.request.IndexRoot;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.diagram.synchronization.IModifiableSynchronizationContext;
import org.simantics.diagram.synchronization.SynchronizationHints;
import org.simantics.diagram.synchronization.graph.GraphSynchronizationHints;
import org.simantics.diagram.synchronization.graph.layer.GraphLayerManager;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.participant.MouseUtil;
import org.simantics.g2d.participant.MouseUtil.MouseInfo;
import org.simantics.modeling.ModelingResources;
import org.simantics.scl.commands.Commands;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.ui.workbench.ResourceEditorInput;
import org.simantics.utils.FileUtils;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

public class ImportSVGPNG {

    @Execute
    public void execute(@Named(IServiceConstants.ACTIVE_PART) MPart part) {
        if (part == null)
            return;
        
        IWorkbenchPart ap;
        Object possibleEditor = part.getObject();
        if (possibleEditor != null && possibleEditor instanceof CompatibilityEditor) {
            CompatibilityEditor editor = (CompatibilityEditor) possibleEditor;
            ap = editor.getPart();
        } else {
            // Fallback for getting active editor part
            ap = WorkbenchUtils.getActiveWorkbenchPart();
        }
        
        IEditorPart viewer = null;
        if (ap instanceof MultiPageEditorPart) {
            MultiPageEditorPart rfe = (MultiPageEditorPart) ap;
            IResourceEditorInput in = (ResourceEditorInput) rfe.getEditorInput();
            final ResourceArray ra = in.getResourceArray();
            ResourceArray symbolEditorInput;
            try {
                symbolEditorInput = SimanticsUI.getSession().syncRequest(new Read<ResourceArray>() {
                    @Override
                    public ResourceArray perform(ReadGraph graph) throws DatabaseException {
                        StructuralResource2 sr = StructuralResource2.getInstance(graph);
                        ModelingResources mr = ModelingResources.getInstance(graph);
                        Resource symbol = graph.getPossibleObject(ra.resources[0], mr.ComponentTypeToSymbol);
                        Resource definedBy = symbol != null ? graph.getPossibleObject(symbol, sr.IsDefinedBy) : null;
                        ResourceArray result = definedBy != null ? new ResourceArray(definedBy) : ResourceArray.EMPTY;
                        return result;
                    }
                });

                IEditorPart[] eps = rfe.findEditors(new ResourceEditorInput("org.simantics.modeling.ui.symbolEditor", symbolEditorInput));
                if (eps.length == 0) {
                    System.out.println("symbol editor part not found from multi page editor part: " + ap);
                    return;
                }
                viewer = eps[0];
            } catch (DatabaseException e) {
                e.printStackTrace();
                return;
            }
        } else if (ap instanceof IEditorPart) {
            viewer = (IEditorPart) ap;
        }
        ICanvasContext ctx = (ICanvasContext) viewer.getAdapter(ICanvasContext.class);
        if (ctx == null) {
            System.out.println("No canvas context");
            return;
        }
        MouseInfo minfo = ctx.getSingleItem(MouseUtil.class).getMousePressedInfo(0);
        if(minfo == null) {
            System.out.println("No mouse info");
            return;
        }
        final Point2D mpos = minfo.canvasPosition;
        IResourceEditorInput input = (IResourceEditorInput)viewer.getEditorInput();
        Resource composite = input.getResource();

        IDiagram idiagram = viewer.getAdapter(IDiagram.class);

        addSVG(mpos.getX(), mpos.getY(), composite, idiagram);
    }

    public static void addSVG(final double mposX, final double mposY, final Resource composite, IDiagram idiagram) {

        Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();

        FileDialog dialog = new FileDialog(shell);
        dialog.setText("Choose an image to be imported");
        dialog.setFilterExtensions(new String[] {"*.svg", "*.png"});

        final String filename = dialog.open();
        if(filename == null)
            return;

        File file = new File(filename);
        try {
            final byte[] data = FileUtils.readFile(file);

            SimanticsUI.getSession().asyncRequest(new WriteRequest() {

                @Override
                public void perform(WriteGraph g) throws DatabaseException {
                    Object svg = Commands.get(g, "Simantics/Diagram/createSVGElementR")
                            .execute(g, g.syncRequest(new IndexRoot(composite)),
                                     composite, suffix(filename), data, mposX, mposY);

                    if (svg != null && svg instanceof Resource) {
                        Resource resource = (Resource) svg;
                        // 7. Put the element on all the currently active layers if possible.
                        IModifiableSynchronizationContext context = idiagram.getHint(SynchronizationHints.CONTEXT);
                        GraphLayerManager glm = context.get(GraphSynchronizationHints.GRAPH_LAYER_MANAGER);
                        if (glm != null) {
                            glm.removeFromAllLayers(g, resource);
                            glm.putElementOnVisibleLayers(idiagram, g, resource);
                        }
                    }
                }

            });
        } catch (FileNotFoundException e) {
            ErrorLogger.defaultLogError(e);
        } catch (IOException e) {
            ErrorLogger.defaultLogError(e);
        }

        return;

    }

    private static String suffix(String fileName) {
        int len = fileName.length();
        if(len < 3) return null;
        else return fileName.substring(len-3,len).toLowerCase();
    }

}
