package org.simantics.modeling.ui.actions;

import java.io.IOException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.ActionFactory;
import org.simantics.graph.compiler.CompilationResult;
import org.simantics.graph.compiler.internal.ltk.Problem;
import org.simantics.modeling.CompilePGraphs;
import org.simantics.modeling.ui.Activator;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Antti Villberg
 */
public class CompilePGraphsAction implements ActionFactory {

    @Override
    public Runnable create(Object target) {
        if (!(target instanceof Resource))
            return null;
        return () -> {
            Job job = new Job("Compile PGraphs") {
                @Override
                protected IStatus run(IProgressMonitor monitor) {
                    try {
                        CompilePGraphs.compilePGraphs((Resource) target, new CompileUserAgent(), monitor);
                        return Status.OK_STATUS;
                    } catch (IOException | DatabaseException e) {
                        ExceptionUtils.logAndShowError(e);
                        return new Status(IStatus.ERROR, Activator.PLUGIN_ID, e.getMessage());
                    } finally {
                        monitor.done();
                    }
                }
            };
            job.schedule();
        };
    }

    public static class CompileUserAgent implements CompilePGraphs.UserAgent {
    	
        @Override
        public void reportProblems(CompilationResult result) {
        	Runnable runnable = () -> {
        		class ErrorMessageDialog extends MessageDialog {
        			public ErrorMessageDialog(Shell shell) {
        				super(shell, 
        						"Problems in the Ontology Definition File", null, 
        						"The following issues were found:", 
        						MessageDialog.ERROR, new String[] { "Continue" }, 0);
        			}

        			@Override
        			protected Control createCustomArea(Composite composite) {
        				GridLayoutFactory.fillDefaults().applyTo(composite);

        				org.eclipse.swt.widgets.List list = new org.eclipse.swt.widgets.List(composite, SWT.BORDER | SWT.READ_ONLY);
        				GridDataFactory.fillDefaults().grab(true, true).applyTo(list);
        				for (Problem problem : result.getErrors())
        					list.add(problem.getLocation() + ": " + problem.getDescription() + "\n");
        				for (Problem problem : result.getWarnings())
        					list.add(problem.getLocation() + ": " + problem.getDescription() + "\n");

        				return composite;
        			}
        		}

        		ErrorMessageDialog md = new ErrorMessageDialog(Display.getCurrent().getActiveShell());
        		md.open();
        	};        	
        	
        	Display display = Display.getCurrent();
        	if (display == null)
        		display = Display.getDefault();
        	display.asyncExec(runnable);
        }

    }

}
