/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.adapters;

import java.util.Map;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.adapters.Removers.ValidationResult;

/**
 * A remover implementation for symbol terminals.
 * 
 * <p>
 * Contrary to ConnectionRelationRemover, this remover will only remove the
 * diagram side connection relations in addition to removing the terminal
 * itself. Deleting a terminal from the symbol does not imply that the whole
 * connection point needs to be removed from the component type configuration,
 * only that the graphical representation of the connection point is removed.
 * 
 * @author Tuukka Lehtonen
 * @see ConnectionRelationRemover
 */
public class TerminalRemover extends ElementRemover {

    public TerminalRemover(Resource terminal) {
        super(terminal);
    }

    @Override
    public String canRemove(ReadGraph graph, Map<Object, Object> aux) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);

        for (Resource diagramConnectionRelation : graph.getObjects(resource, DIA.HasConnectionPoint)) {
            for (Resource connectionRelation : graph.getObjects(diagramConnectionRelation, MOD.DiagramConnectionRelationToConnectionRelation)) {
                ValidationResult result = Removers.validateConnectionRelationRemoval(graph, connectionRelation, diagramConnectionRelation);
                if (result.inUse())
                    return Removers.formatError(graph, result);
            }
        }

        return null;
    }

    @Override
    public void remove(WriteGraph graph) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);

        for (Resource diagramConnectionPoint : graph.getObjects(resource, DIA.HasConnectionPoint)) {
            Resource diagramConnectionPointInverse = graph.getPossibleObject(diagramConnectionPoint, L0.InverseOf);
            graph.deny(diagramConnectionPoint);
            if (diagramConnectionPointInverse != null)
                graph.deny(diagramConnectionPointInverse);
        }

        removeElement(graph);
    }

}
