/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.adapters;

import gnu.trove.set.hash.THashSet;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Set;

import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Remover;
import org.simantics.db.layer0.adapter.impl.AbstractRemover;
import org.simantics.db.layer0.adapter.impl.EntityRemover;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.diagram.content.ConnectionUtil;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.utils.binaryPredicates.OrderedSetElementsPredicate;
import org.simantics.modeling.ModelingResources;
import org.simantics.structural.stubs.StructuralResource2;

/**
 * A {@link Remover} implementation for removing DIA.Element instances.
 * 
 * The implementation has separate paths for STR.Connection instances and other
 * DIA.Element instances. For DIA.Element this remover will:
 * <ol>
 * <li>List connections/monitors the element is attached to</li>
 * <li>Disconnect the element from its owner diagram(s)</li>
 * <li>Remove the element itself with EntityRemover default logic</li>
 * <li>Recursively remove all related degenerate connections, i.e. connections
 * that only have one connector after remove the one connected to this removed
 * element.</li>
 * <li>Recursively remove monitors attached to the removed element</li>
 * </ol>
 * 
 * <p>
 * For STR.Connections this remover will:
 * <ol>
 * <li>List all related data (monitors, mapped data)</li> 
 * <li>Remove the connection itself</li>
 * <li>Remove the related data</li>
 * </ol>
 * 
 * @author Tuukka Lehtonen
 */
public class ElementRemover extends AbstractRemover {

    private static final boolean DEBUG = false;

    public ElementRemover(Resource element) {
        super(element);
    }

    @Override
    public void remove(WriteGraph graph) throws DatabaseException {
        // 0. Find all connections to the removed element
        StructuralResource2 STR = StructuralResource2.getInstance(graph);

        if (graph.isInstanceOf(resource, STR.Connection)) {
            removeConnection(graph);
        } else {
            removeElement(graph);
        }
    }

    public void removeElement(WriteGraph graph) throws DatabaseException {
        if (DEBUG)
            System.out.println(this + " removing element");

        Set<Resource> connectors = null;

        StructuralResource2 STR = StructuralResource2.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);

        Collection<Resource> connectedTo = graph.getObjects(resource, STR.IsConnectedTo);
        for (Resource connector : connectedTo) {
            if (connectors == null)
                connectors = new THashSet<Resource>(connectedTo.size());
            connectors.add(connector);
        }
        Collection<Resource> monitors = graph.getObjects(resource, DIA.HasMonitorComponent_Inverse);

        if (DEBUG) {
            if (connectors != null)
                System.out.println(this + " found " + connectors.size() + " connectors connected to element");
            if (!monitors.isEmpty())
                System.out.println(this + " found " + monitors.size() + " monitors attached to element");
        }

        // 1. Disconnect element from diagrams
        for (Resource diagram : OrderedSetElementsPredicate.INSTANCE.getSubjects(graph, resource)) {
            OrderedSetUtils.remove(graph, diagram, resource);
        }

        // 2. Delete element itself
        EntityRemover.remove(graph, resource);

        // 3. Recursively remove all related degenerate connections
        // i.e. connections that only have one connector after remove the
        // one connected to this removed element.
        ConnectionUtil cu = null;
        Set<Resource> connectionsToRemove = null;
        if (connectors != null) {
            cu = new ConnectionUtil(graph);
            Set<Resource> touchedConnections = new THashSet<Resource>(connectors.size());
            for (Resource connector : connectors) {
                Resource connection = ConnectionUtil.tryGetConnection(graph, connector);
                if (connection != null)
                    touchedConnections.add(connection);
                cu.removeConnectionPart(connector);
            }
            for (Resource connection : touchedConnections) {
                int removedConnectors = cu.removeUnusedConnectors(connection);
                if (DEBUG)
                    System.out.println(this + " PRUNED " + removedConnectors + " UNUSED CONNECTORS FROM TOUCHED CONNECTION " + connection);
                while (true) {
                    int removedInteriorRouteNodes = cu.removeExtraInteriorRouteNodes(connection);
                    if (DEBUG)
                        System.out.println(this + " PRUNED " + removedInteriorRouteNodes + " INTERIOR ROUTE NODES FROM TOUCHED CONNECTION " + connection);
                    if (removedInteriorRouteNodes == 0)
                        break;
                }
                int connectorCount = cu.getConnectedConnectors(connection, null).size();
                if (DEBUG)
                    System.out.println(this + " \t" + connectorCount + " CONNECTORS LEFT");
                if (connectorCount < 2) {
                    if (connectionsToRemove == null)
                        connectionsToRemove = new THashSet<Resource>(touchedConnections.size());
                    connectionsToRemove.add(connection);
                }
            }
        }
        if (connectionsToRemove != null) {
            for (Resource connection : connectionsToRemove) {
                if (DEBUG)
                    System.out.println(this + " REMOVING connection " + connection);
                RemoverUtil.remove(graph, connection);
            }
        }

        // 4. Recursively remove monitors attached to element
        if (!monitors.isEmpty())
            for (Resource monitor : monitors)
                RemoverUtil.remove(graph, monitor);

    }

    public void removeConnection(WriteGraph graph) throws DatabaseException {
        if (DEBUG)
            System.out.println(this + " removing connection element");

        DiagramResource DIA = DiagramResource.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);

        // 0. Get set of related data
        Collection<Resource> objects = new ArrayList<Resource>();
        for (Resource relation : new Resource[] {
                DIA.HasMonitorComponent_Inverse,
                MOD.ElementToComponent,
                MOD.DiagramConnectionToConnection,
                MOD.DiagramConnectionToConnectionSpecial
        }) {
            for (Resource object : graph.getObjects(resource, relation)) {
                if (!object.equals(resource))
                    objects.add(object);
            }
        }

        // 1. Remove connection itself
        new ConnectionUtil(graph).removeConnection(resource);

        // 2. Recursively remove related data
        if (!objects.isEmpty())
            for (Resource object : objects)
                RemoverUtil.remove(graph, object);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + resource;
    }

}
