package org.simantics.modeling;

import gnu.trove.map.hash.THashMap;

import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.ByteType;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.LongType;
import org.simantics.databoard.type.StringType;
import org.simantics.db.layer0.request.PropertyInfo;
import org.simantics.scl.compiler.types.TCon;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.kinds.Kinds;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCLTypeUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(SCLTypeUtils.class);
    private static final THashMap<String, Type> TYPE_MAP = new THashMap<String, Type>();
    
    private static void add(TCon type) {
        TYPE_MAP.put(type.name, type);
    }
    
    static {
        add(Types.BOOLEAN);
        add(Types.BYTE);
        add(Types.CHARACTER);
        add(Types.SHORT);
        add(Types.INTEGER);
        add(Types.LONG);
        add(Types.FLOAT);
        add(Types.DOUBLE);
        add(Types.STRING);
        
        TYPE_MAP.put("[Boolean]", Types.list(Types.BOOLEAN));
        TYPE_MAP.put("[Byte]", Types.list(Types.BYTE));
        TYPE_MAP.put("[Character]", Types.list(Types.CHARACTER));
        TYPE_MAP.put("[Short]", Types.list(Types.SHORT));
        TYPE_MAP.put("[Integer]", Types.list(Types.INTEGER));
        TYPE_MAP.put("[Long]", Types.list(Types.LONG));
        TYPE_MAP.put("[Float]", Types.list(Types.FLOAT));
        TYPE_MAP.put("[Double]", Types.list(Types.DOUBLE));
        TYPE_MAP.put("[String]", Types.list(Types.STRING));
        
        TYPE_MAP.put("Vector Boolean", Types.vector(Types.BOOLEAN));
        TYPE_MAP.put("Vector Byte", Types.vector(Types.BYTE));
        TYPE_MAP.put("Vector Character", Types.vector(Types.CHARACTER));
        TYPE_MAP.put("Vector Short", Types.vector(Types.SHORT));
        TYPE_MAP.put("Vector Integer", Types.vector(Types.INTEGER));
        TYPE_MAP.put("Vector Long", Types.vector(Types.LONG));
        TYPE_MAP.put("Vector Float", Types.vector(Types.FLOAT));
        TYPE_MAP.put("Vector Double", Types.vector(Types.DOUBLE));
        TYPE_MAP.put("Vector String", Types.vector(Types.STRING));
        TYPE_MAP.put("ByteArray", Types.BYTE_ARRAY);
        
        add((TCon)Types.RESOURCE);
        add(Types.con("Simantics/GUID", "GUID")); // L0.GUID
        add(Types.con("Simantics/Variables", "StructuredProperty")); // L0.methods
        add(Types.con("Simantics/Variables", "ValueAccessor")); // L0.ValueAccessor
        add(Types.con("Simantics/Variables", "VariableMap"));
    }
    
    /**
     * This is very rude method for converting SCL type text to SCL type.
     * All uses of this method should be replaced in the future by use of SCL compiler
     * and actual lookups to dependent SCL modules 
     */
    public static Type getType(String typeText) {
        Type type = TYPE_MAP.get(typeText);
        if(type == null) {
            LOGGER.warn("SCLTypeUtils.getType cannot transform '" + typeText + "' to type. Returns a as default.");
            return Types.var(Kinds.STAR);
        }
        return type;
    }
    
    public static Type getType(Datatype dataType) {
        if(dataType instanceof DoubleType)
            return Types.DOUBLE;
        else if(dataType instanceof IntegerType)
            return Types.INTEGER;
        else if(dataType instanceof StringType)
            return Types.STRING;
        else if(dataType instanceof BooleanType)
            return Types.BOOLEAN;
        else if(dataType instanceof FloatType)
            return Types.FLOAT;
        else if(dataType instanceof LongType)
            return Types.LONG;
        else if(dataType instanceof ByteType)
            return Types.BYTE;
        else if(dataType instanceof ArrayType)
            return Types.list(getType(((ArrayType)dataType).componentType));
        else {
            LOGGER.warn("SCLTypeUtils.getType cannot transform data type '" + dataType + "' to type. Returns a as default.");
            return Types.var(Kinds.STAR);
        }
    }
    
    public static Type getType(PropertyInfo propertyInfo) {
        if(propertyInfo.requiredValueType != null)
            return getType(propertyInfo.requiredValueType);
        else if(propertyInfo.requiredDatatype != null)
            return getType(propertyInfo.requiredDatatype);
        else {
            LOGGER.warn(propertyInfo.name + " doesn't have type information. Returns a as default.");
            return Types.var(Kinds.STAR);
        }
    }
}
