/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.layer0.utils.binaryPredicates;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.simantics.db.Resource;
import org.simantics.db.ReadGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.utils.datastructures.Pair;

public class CompositePredicate extends BinaryPredicate {
	IBinaryPredicate first;
	IBinaryPredicate second;
	
	int hasStatementDirection;
	int getStatementsDirection;
	
	public CompositePredicate(IBinaryPredicate first, IBinaryPredicate second) {
		this.first = first;
		this.second = second;		
		if(first.supportsGetStatements() && second.supportsGetObjects())
			getStatementsDirection = 1;
		else if(second.supportsGetStatements() && first.supportsGetSubjects())
			getStatementsDirection = 2;
		else
			this.getStatementsDirection = 0;
		if(first.supportsGetObjects())
			hasStatementDirection = 1;
		else if(second.supportsGetSubjects())
			hasStatementDirection = 2;
		else
			throw new IllegalArgumentException("Cannot compose binary predicates such that the first one does not support getObjects and the second one does not support getSubjects.");
	}

	@Override
	public Collection<Resource> getObjects(ReadGraph g, Resource subject) throws DatabaseException {
		Set<Resource> result = new HashSet<Resource>();
		for(Resource r : first.getObjects(g, subject))
			result.addAll(second.getObjects(g, r));
		return result;
	}

	@Override
	public Collection<Pair<Resource, Resource>> getStatements(ReadGraph g) throws DatabaseException {
		Set<Pair<Resource, Resource>> result = 
			new HashSet<Pair<Resource, Resource>>();
		if(getStatementsDirection == 1)
			for(Pair<Resource, Resource> p : first.getStatements(g))
				for(Resource r : second.getObjects(g, p.second))
					result.add(new Pair<Resource, Resource>(p.first, r));
		else if(getStatementsDirection == 2)
			for(Pair<Resource, Resource> p : second.getStatements(g))
				for(Resource r : second.getSubjects(g, p.first))
					result.add(new Pair<Resource, Resource>(r, p.second));
		else
			throw new UnsupportedOperationException();
		return result;
	}

	@Override
	public Collection<Resource> getSubjects(ReadGraph g, Resource object) throws DatabaseException {
		Set<Resource> result = new HashSet<Resource>();
		for(Resource r : second.getSubjects(g, object))
			result.addAll(first.getSubjects(g, r));
		return result;
	}

	@Override
	public boolean has(ReadGraph g, Resource subject, Resource object) throws DatabaseException {
		if(hasStatementDirection == 1) {
			for(Resource r : first.getObjects(g, subject))
				if(second.has(g, r, object))
					return true;
		}
		else {
			for(Resource r : second.getSubjects(g, object))
				if(second.has(g, subject, r))
					return true;
		}
		return false;
	}

	@Override
	public void add(WriteGraph g, Resource subject, Resource object) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void remove(WriteGraph g, Resource subject, Resource object) {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean supportsAdditions() {
		return false;
	}

	@Override
	public boolean supportsGetObjects() {
		return first.supportsGetObjects() && second.supportsGetObjects();
	}

	@Override
	public boolean supportsGetStatements() {
		return getStatementsDirection != 0;
	}

	@Override
	public boolean supportsGetSubjects() {
		return first.supportsGetSubjects() && second.supportsGetSubjects();
	}

	@Override
	public boolean supportsRemovals() {		
		return false;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((first == null) ? 0 : first.hashCode());
		result = prime * result + ((second == null) ? 0 : second.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		CompositePredicate other = (CompositePredicate) obj;
		if (first == null) {
			if (other.first != null)
				return false;
		} else if (!first.equals(other.first))
			return false;
		if (second == null) {
			if (other.second != null)
				return false;
		} else if (!second.equals(other.second))
			return false;
		return true;
	}	
	
}
