/*******************************************************************************
 * Copyright (c) 2007, 2017 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - #6948
 *******************************************************************************/
package org.simantics.issues.ui;

import java.util.Collections;
import java.util.Map;

import org.eclipse.jface.resource.ImageDescriptor;
import org.simantics.browsing.ui.common.ColumnKeys;
import org.simantics.browsing.ui.model.images.ImageRule;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.silk.SilkResources;

/**
 * @author Tuukka Lehtonen
 */
public class IssueImageRule implements ImageRule {

    private final String DESCRIPTION = ColumnKeys.SINGLE;

    private final ImageDescriptor tick;
    private final ImageDescriptor fatal;
    private final ImageDescriptor error;
    private final ImageDescriptor warning;
    private final ImageDescriptor info;
    private final ImageDescriptor note;
    private final ImageDescriptor help;

    public IssueImageRule(ReadGraph graph) throws DatabaseException {
        SilkResources SILK = SilkResources.getInstance(graph);
        tick = graph.adapt(SILK.tick, ImageDescriptor.class);
        fatal = graph.adapt(SILK.bomb, ImageDescriptor.class);
        error = graph.adapt(SILK.exclamation, ImageDescriptor.class);
        warning = graph.adapt(SILK.error, ImageDescriptor.class);
        info = graph.adapt(SILK.information, ImageDescriptor.class);
        note = graph.adapt(SILK.note, ImageDescriptor.class);
        help = graph.adapt(SILK.help, ImageDescriptor.class);
    }

    @Override
    public boolean isCompatible(Class<?> contentType) {
        return contentType.equals(Variable.class);
    }

    @Override
    public Map<String, ImageDescriptor> getImage(ReadGraph graph, Object content) throws DatabaseException {
        Variable issue = (Variable) content;
        String severity = issue.getPossiblePropertyValue(graph, "severity");
        if (severity == null)
            return Collections.emptyMap();
        boolean resolved = isResolved(graph, issue);
        return Collections.singletonMap(DESCRIPTION, resolved ? tick : toImageDescriptor(severity));
    }

    public boolean isResolved(ReadGraph graph, Variable issue) throws DatabaseException {
        Resource issueResource = issue.getPossibleRepresents(graph);
        if (issueResource != null)
            return graph.hasStatement(issueResource, IssueResource.getInstance(graph).Resolved);

        Boolean resolved = issue.getPossiblePropertyValue(graph, "resolved");
        return Boolean.TRUE.equals(resolved);
    }

    private ImageDescriptor toImageDescriptor(String severity) {
        switch (severity) {
        case "Fatal":   return fatal;
        case "Error":   return error;
        case "Warning": return warning;
        case "Info":    return info;
        case "Note":    return note;
        default:        return help;
        }
    }

}
