/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.issues.common;

import gnu.trove.map.hash.TObjectByteHashMap;
import gnu.trove.set.hash.THashSet;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.issues.common.preferences.IssuePrefs;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.layer0.Layer0;
import org.simantics.operation.Layer0X;
import org.simantics.simulation.ontology.SimulationResource;

/**
 * @author Tuukka Lehtonen
 */
public class AllVisibleIssues extends BinaryRead<Resource, Boolean, Set<Variable>> {

    public AllVisibleIssues(Resource project) {
        super(project, Boolean.FALSE);
    }

    public AllVisibleIssues(Resource project, boolean onlyUnresolved) {
        super(project, onlyUnresolved);
    }

    @Override
    public Set<Variable> perform(ReadGraph graph) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        Layer0X L0X = Layer0X.getInstance(graph);
        IssueResource ISSUE = IssueResource.getInstance(graph);
        SimulationResource SIMU = SimulationResource.getInstance(graph);

        Resource project = Simantics.getProjectResource();
        boolean showHidden = false;
        boolean showNormal = true;
        boolean showUser = true;
        if (project != null) {
            showHidden = IssuePrefs.showHiddenIssues(graph, project);
            showNormal = IssuePrefs.showNormalIssues(graph, project);
            showUser = IssuePrefs.showUserIssues(graph, project);
        }

        Instances issueIndex = graph.getPossibleAdapter(ISSUE.Issue, Instances.class);

        // Cache for source activeness.
        // 0 == not in cache, 1 == false, 2 == true
        TObjectByteHashMap<Resource> sourceActivenessCache = new TObjectByteHashMap<Resource>();

        Set<Variable> result = new THashSet<Variable>(1013);

        for (Resource model : graph.syncRequest(new ObjectsWithType(parameter, L0X.Activates, SIMU.Model))) {
            Collection<Resource> modelIssues = graph.syncRequest(new ObjectsWithType(model, L0.ConsistsOf, ISSUE.Issue));
            Collection<Resource> indexedIssues = issueIndex != null ? issueIndex.find(graph, model) : Collections.<Resource>emptyList();
            Collection<Resource> issues = !indexedIssues.isEmpty() ? new THashSet<Resource>(modelIssues.size() + indexedIssues.size()) : modelIssues;
            if (!indexedIssues.isEmpty()) {
                issues.addAll(modelIssues);
                issues.addAll(indexedIssues);
            }

            for (Resource issue : issues) {
                // Filter out unwanted material
                boolean resolved = graph.hasStatement(issue, ISSUE.Resolved);
                if (parameter2 && resolved)
                    continue;
                boolean hidden = graph.hasStatement(issue, ISSUE.Hidden);
                boolean user = graph.hasStatement(issue, ISSUE.UserIssue);
                boolean normal = !hidden && !user;
                if (!showHidden && hidden)
                    continue;
                if (!showUser && user)
                    continue;
                if (!showNormal && normal)
                    continue;

                Resource source = graph.getPossibleObject(issue, ISSUE.IssueSource_Manages_Inverse);
                if (source != null) {
                    byte cache = sourceActivenessCache.get(source);
                    boolean active = cache == 2 ? true : false;
                    if (cache == 0) {
                        active = Boolean.TRUE.equals(graph.getPossibleRelatedValue(source, ISSUE.IssueSource_active));
                        sourceActivenessCache.put(source, active ? (byte) 2 : (byte) 1);
                    }
                    if (!active)
                        continue;
                }

                Variable var = Variables.getPossibleVariable(graph, issue);
                if (var != null)
                    result.add(var);
            }
        }

        // System.out.println("AllActiveIssues returned " + result.size());
        return result;
    }

}
