package org.simantics.history;

import java.util.Arrays;

import org.simantics.databoard.accessor.StreamAccessor;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.history.util.StreamIterator;

public class HistorySamplerItem2 implements Comparable<HistorySamplerItem2> {

	public static class LevelItem implements Comparable<LevelItem> {
		public final String id;
		public final double samplingInterval;

		public LevelItem(String id, double samplingInterval) {
			this.id = id;
			this.samplingInterval = samplingInterval;
		}

		@Override
		public int compareTo(LevelItem o) {
			int i = id.compareTo(o.id);
			return i != 0 ? i : Double.compare(samplingInterval, o.samplingInterval);
		}
	}

	Collector collector;
	HistoryManager history;         // History source for this item
	LevelItem[] items;

	// State data
	StreamAccessor accessor;			// Stream accessor
	public StreamIterator iter;
	public int chartDataId;

	public HistorySamplerItem2(Collector collector, HistoryManager history, LevelItem[] items, int identityHashCode) {
		if (items.length == 0)
			throw new IllegalArgumentException("Must have at least one existing history item to sample, zero provided");
		this.collector = collector;
		this.history = history;
		this.items = items;
		this.chartDataId = identityHashCode;
	}

	public void open() throws HistoryException {
		accessor = history.openStream(items[0].id, "r");
		iter = new StreamIterator( accessor );
	}

	public void open(double pixelsPerSecond) throws HistoryException {
		LevelItem f = getFormat(pixelsPerSecond);
		accessor = history.openStream(f.id, "r");
		iter = new StreamIterator( accessor );
	}

	public void close() {
		if (accessor!=null) {
			try {
				accessor.close();
			} catch (AccessorException e) {
			}
		}
		accessor = null;
		iter = null;
	}

	@Override
	public int compareTo(HistorySamplerItem2 o) {
		int m = Math.min(items.length, o.items.length);
		for (int j = 0; j < m; ++j) {
			int i = items[j].compareTo(o.items[j]);
			if (i != 0)
				return i;
		}
		return 0;
	}

	@Override
	public int hashCode() {
		int code = 0x2304;
		code = 13*code + Arrays.hashCode(items);
		code = 13*code + history.hashCode();
		return code;
	}

	@Override
	public boolean equals(Object obj) {
		if ( obj == null ) return false;
		if ( obj instanceof HistorySamplerItem2 == false ) return false;
		HistorySamplerItem2 other = (HistorySamplerItem2) obj;
		if ( !other.history.equals(history) ) return false;
		if ( !Arrays.equals(other.items, items) ) return false;
		return true;
	}

	private LevelItem getFormat(double secondsPerPixel) throws HistoryException {
		LevelItem result = null;
		for (LevelItem format : items) {
			double interval = format.samplingInterval;
			if (Double.isNaN( interval ) || interval <= secondsPerPixel) {
				result = format;
			} else {
				break;
			}
		}
		return result;
	}

}
