package org.simantics.g2d.element.impl;

import java.util.Map;

import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.ElementAdapter;
import org.simantics.g2d.element.handler.LifeCycle;
import org.simantics.utils.datastructures.hints.HintContext;
import org.simantics.utils.datastructures.hints.HintStack;

/**
 * @author Tuukka Lehtonen
 */
public class MutatedElement extends HintStack implements IElement {

    IDiagram     diagram;
    IElement     orig;
    ElementClass origClass;
    HintContext  mutations;

    public MutatedElement(IElement orig) {
        if (orig == null)
            throw new IllegalArgumentException("null arg");
        this.orig = orig;
        this.origClass = orig.getElementClass();
        this.addHintContext(orig, 0);
    }

    @Override
    public void addedToDiagram(IDiagram ctx) {
        this.diagram = ctx;
    }

    @Override
    public IDiagram getDiagram() {
        assert diagram != null;
        return diagram;
    }

    @Override
    public IDiagram peekDiagram() {
        return diagram;
    }

    @Override
    public ElementClass getElementClass() {
        return origClass;
    }

    @Override
    public synchronized void clearWithoutNotification() {
        if (mutations != null)
            mutations.clearWithoutNotification();
    }

    @Override
    public synchronized <E> E removeHint(Key key) {
        if (key == null)
            throw new IllegalArgumentException("key is null");
        if (mutations == null)
            return null;
        E result = mutations.removeHint(key);
        if (mutations.size() == 0) {
            removeHintContext(mutations);
            mutations = null;
        }
        return result;
    }

    @Override
    public synchronized void setHint(Key key, Object value) {
        if (key == null)
            throw new IllegalArgumentException("key is null");
        if (value == null)
            throw new IllegalArgumentException("value is null");
        if (!key.isValueAccepted(value))
            throw new RuntimeException("Value \"" + value + "\" is not accepted with key " + key.getClass().getName());
        if (mutations == null) {
            mutations = new HintContext();
            addHintContext(mutations, 10);
        }
        mutations.setHint(key, value);
    }

    @Override
    public synchronized void setHints(Map<Key, Object> hints) {
        if (mutations == null) {
            mutations = new HintContext();
            addHintContext(mutations, 10);
        }
        mutations.setHints(hints);
    }

    @Override
    public void destroy() {
        for (LifeCycle lc : getElementClass().getItemsByClass(LifeCycle.class))
            lc.onElementDestroyed(this);
        dispose();
    }

    @Override
    public void dispose() {
        if (mutations != null)
            this.removeHintContext(mutations);
        this.removeHintContext(orig);
        orig = null;
        origClass = null;
        mutations = null;
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    public Object getAdapter(Class adapter) {
        for (ElementAdapter ea : getElementClass().getItemsByClass(ElementAdapter.class)) {
            Object result = ea.adapt(this, adapter);
            if (result != null)
                return result;
        }
        return null;
    }

    @Override
    public int hashCode() {
        return orig != null ? orig.hashCode() : 0;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        return orig == null ? false : orig.equals(obj);
    }

    @Override
    public String toString() {
        return "MutatedElement[" + orig + " " + origClass + "]";
    }

}
