/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.element.handler.impl;

import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.simantics.g2d.diagram.handler.Topology.Terminal;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.TerminalLayout;
import org.simantics.g2d.element.handler.TerminalTopology;
import org.simantics.g2d.utils.geom.DirectionSet;
import org.simantics.utils.datastructures.ValueUtils;

/**
 * 
 * 
 * Example:
 * 
 *   TerminalHandler th = 
 *   	StaticTerminals.compile(
 *   		topTerminal,	0.0, -10.0, DirectionSet.N,
 *   		bottomTerminal,	0.0,  10.0, DirectionSet.S,
 *        	leftTerminal,  -10.0,  0.0, DirectionSet.W,
 *          rightTerminal,  20.0,  0.0, DirectionSet.E);
 * 
 * @author Toni Kalajainen
 */
public class StaticTerminals implements TerminalLayout, TerminalTopology {

    private static final long serialVersionUID = -7324592260579191478L;

    /**
	 * Compile terminal handler with static terminals.
	 * The xytArray is an varargs array which consists of sequences of
	 *  (terminal, x-position, y-position, DirectionSet).
	 * 
	 * @param xytArray an array of (x, y, terminal) triples
	 * @return terminal reference
	 */
	public static TerminalLayout compile(Object ... xytArray)
	{
		return new StaticTerminals(xytArray);
	}
	
	private class TerminalImpl implements Terminal {
		double x, y;
		DirectionSet ds;
		Terminal t;
	}
	
	Map<Terminal, TerminalImpl> terminalMap = new HashMap<Terminal, TerminalImpl>();
	ArrayList<Terminal> terminals = new ArrayList<Terminal>();
	
	StaticTerminals(Object ... xytArray)
	{
		assert(xytArray.length % 4 == 0);
		for (int i=0; i<xytArray.length/4; i++)
		{
			TerminalImpl ti = new TerminalImpl();
			ti.t = (Terminal) xytArray[i*4+0];
			ti.x = ValueUtils.toDoubleScalar( xytArray[i*4+1] );
			ti.y = ValueUtils.toDoubleScalar( xytArray[i*4+2] );
			DirectionSet ds = (DirectionSet) xytArray[i*4+3];
			ti.ds = ds;
			terminals.add(ti.t);
			terminalMap.put(ti.t, ti);
		}
	}	

	@Override
	public AffineTransform getTerminalPosition(IElement node, Terminal t) {
		TerminalImpl ti = terminalMap.get(t);
		if (ti==null) return null;
		AffineTransform result = new AffineTransform();
		result.setToTranslation(ti.x, ti.y);
		return result;
	}

	@Override
	public void getTerminals(IElement e, Collection<Terminal> result) {
		result.addAll(terminals);
	}

	@Override
	public boolean getTerminalDirection(IElement node, Terminal t, DirectionSet directions) {
		TerminalImpl ti = terminalMap.get(t);
		if (ti==null) return false;
		directions.addAll(ti.ds);
		return true;
	}

	@Override
	public Shape getTerminalShape(IElement node, Terminal t) {
		return null;
	}
	
}
