package org.simantics.export.ui;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.nebula.widgets.grid.Grid;
import org.eclipse.nebula.widgets.grid.GridColumn;
import org.eclipse.nebula.widgets.grid.GridItem;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Composite;
import org.simantics.db.exception.DatabaseException;
import org.simantics.export.core.ExportContext;
import org.simantics.export.core.error.ExportException;
import org.simantics.export.core.intf.ContentType;
import org.simantics.export.core.intf.Discoverer;
import org.simantics.export.core.intf.Exporter;
import org.simantics.export.core.intf.Format;
import org.simantics.export.core.intf.Publisher;
import org.simantics.export.core.manager.Content;
import org.simantics.export.core.manager.ExportWizardResult;
import org.simantics.export.core.util.ExportQueries;
import org.simantics.export.ui.util.ExportUIQueries;
import org.simantics.utils.datastructures.MapList;
import org.simantics.utils.datastructures.ToStringComparator;
import org.simantics.utils.strings.AlphanumComparator;
import org.simantics.utils.ui.workbench.StringMemento;

/**
 * Show wizard page where content and export format is selected.
 *
 * @author toni.kalajainen@semantum.fi
 */
public class ContentSelectionPage extends WizardPage {
	
	/** Key for preference setting that contains sub-mementos for each content URI. */
	public static final String KEY_FORMAT_SELECTIONS = "org.simantics.modeling.ui.export.wizard.formatSelections";

	// UI stuff
	LocalResourceManager resourceManager;
	Grid grid;
	
	// Initial data
	ExportContext ctx; 
	// Hash-code for the initial data
	String initialSelectionKey;
	Set<Content> initialSelection = new HashSet<Content>();
		
	// Previous selections
	StringMemento formatSelections;
	
	// Initializations
	Collection<String> allModels, selectedModels;
	ToStringComparator toStringComparator = new ToStringComparator();
	MapList<ContentType, String> content; // ContentType -> Content[]
	MapList<ContentType, Format> typeToFormatMap; // ContentType -> Format
	MapList<String, ContentType> contentToTypeMap; // uri -> ContentType
	Map<String, Map<String, String>> labels; // ContentType -> URI -> Label
	MapList<String, String> modelContent; // ModelURI -> ContentURI
	
	// User selections
	List<Content> contentSelection = new ArrayList<Content>();
	
	public ContentSelectionPage(ExportContext ctx) throws ExportException {
		super("Select Content", "Select the PDF Pages and the attachments", null);
		this.ctx = ctx;
		
		init();
	}
	
	void init() throws ExportException {
		try {
			System.out.println("Found Content Types:");
			for ( ContentType ct : ctx.eep.contentTypes() ) {
				System.out.println("   "+ct);
			}
			System.out.println();
			
			System.out.println("Exporters:");
			for ( Exporter ex : ctx.eep.exporters() ) {
				System.out.println("   "+ex);
			}
			System.out.println();

			System.out.println("Formats:");
			for ( Format format : ctx.eep.formats() ) {
				System.out.println("   "+format);
			}
			System.out.println();

			System.out.println("Discoverers:");
			for ( Discoverer discoverer : ctx.eep.discoverers() ) {
				System.out.println("   "+discoverer);
			}
			System.out.println();
			
			System.out.println("Publishers:");
			for ( Publisher publisher : ctx.eep.publishers() ) {
				System.out.println("   "+publisher.id());
			}
			System.out.println();
			
			// Organize formats by content types - Filter out ContentTypes that don't have exporter and format.
			System.out.println("Mapped ContentTypes to Exporters:");
			typeToFormatMap = MapList.use( new TreeMap<ContentType, List<Format>>(toStringComparator) );
			for ( ContentType ct : ctx.eep.contentTypes() ) {
				for ( Exporter exp : ctx.eep.getExportersForContentType( ct.id() ) ) {
					Format format = ctx.eep.getFormat( exp.formatId() );
					if ( format==null ) continue;
					System.out.println("    "+ct.id()+" -> "+format.fileext());
					if (!typeToFormatMap.contains(ct, format)) typeToFormatMap.add(ct, format);
				}
			}
			System.out.println();
			
			// Discover the models in the project
			allModels = ctx.session.syncRequest( ExportUIQueries.models(ctx.project) );

			// Calculate hash for the initial selection
			int initialContentHash = 0x52f3a45;
			for ( String content : ctx.selection ) {
				initialContentHash = 13*initialContentHash + content.hashCode();
			}
			initialSelectionKey = "InitialSelection-"+initialContentHash;
			String sel = ctx.store.get(initialSelectionKey, null);
			if ( sel != null ) {
				initialSelection = ExportWizardResult.parse(sel);
			} else {
				// First time wizard was called with this selection.
				// Check in
				for ( String contentUri : ctx.selection ) {
					initialSelection.add( new Content(contentUri, null, "all", null, null, null) );
				}
			}
			
			// Choose the models from user interface selection
			selectedModels = new ArrayList<String>();
			StringBuilder modelsStr = new StringBuilder(); 
			for ( String content : ctx.selection ) {
				for ( String model : allModels ) {
					if ( content.equals(model) || content.startsWith(model + "/") ) {
						if ( !selectedModels.contains( model ) ) {
							selectedModels.add( model );
							if ( modelsStr.length()>0 ) modelsStr.append(", ");
							modelsStr.append( model );
						}
					}
				}
			}
			// If user has nothing selected, choose active models
			if ( selectedModels.isEmpty() ) selectedModels.addAll( ctx.activeModels );
			// If there are no active models, choose all models
			if ( selectedModels.isEmpty() ) selectedModels.addAll( allModels );
			// UI labels
			labels = new HashMap<String, Map<String, String>>();			
			labels.put( "model", ctx.session.syncRequest( ExportQueries.labels( selectedModels ) ) ); // Should Model CT be used for labeling? 
			
			// Discover contents
			System.out.println("Discovering content: "+modelsStr);
			content = MapList.use( new TreeMap<ContentType, List<String>>(toStringComparator) );
			contentToTypeMap = MapList.use( new TreeMap<String, List<ContentType>>(toStringComparator) );
			modelContent = MapList.use( new TreeMap<String, List<String>>() );
			
			for ( ContentType ct : typeToFormatMap.getKeys() ) {
				System.out.println("    "+ct.label());
				for ( Discoverer discoverer : ctx.eep.getDiscoverers( ct.id() )) {
					System.out.println("         "+discoverer.toString());
					
					// Get content Uris
					Collection<String> contents = discoverer.discoverContent(ctx, selectedModels);
					List<String> contentUris = new ArrayList<String>( contents );

					// Get UI Labels
					Map<String, String> lbls = ct.getLabels(ctx, contentUris); 
					labels.put( ct.id(), lbls );
					
					// Sort content
					IndirectComparator comp = new IndirectComparator();
					comp.labels = lbls;
					Collections.sort( contentUris, comp );
					
					for ( String contentId : contentUris ) {
						content.add( ct, contentId );
						contentToTypeMap.add(contentId, ct);
						//modelContent.add(key)
						System.out.println("            "+contentId);
					}
					
				}
			}
			System.out.println();
						
				
		} catch (DatabaseException e) {
			throw new ExportException(e);
		}

		
	}
	
	@Override
	public void createControl(Composite parent) {
	    grid = new Grid(parent, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL | SWT.MULTI);
	    grid.setHeaderVisible(true);
	    
	    resourceManager = new LocalResourceManager(JFaceResources.getResources(), grid);
	    Color contentTypeColor = resourceManager.createColor( new RGB(245, 245, 252) );
	    GridColumn column = new GridColumn(grid,SWT.NONE);
	    column.setTree(true);
	    column.setText("Name");
	    column.setWidth(200);	    

	    // "Pagees"
	    assertColumnIndex(1);
	    	    
	    Format pdfFormat = ctx.eep.getFormat("pdf");
	    
	    
	    ImageDescriptor folderID = null;
		try {
			URL folderUrl = new URL("platform:/plugin/com.famfamfam.silk/icons/folder.png");
		    folderID = ImageDescriptor.createFromURL( folderUrl );
		} catch (MalformedURLException e) {
			e.printStackTrace();
		}
		
		List<GridItem> selectedNodes = new ArrayList<GridItem>();
	    
	    // Iterate all models
	    for ( String modelUri : contentToTypeMap.getKeys() ) {
	    	ContentType modelContentType = null;
	    	for ( ContentType ct : contentToTypeMap.getValues(modelUri) ) {
	    		if ( ct.isModel() ) {
	    			modelContentType = ct;
	    			break;
	    		}
	    	}
	    	if (modelContentType==null) continue;
	    	
	    	// Create Model Node
	    	String modelLabel = labels.get("model").get(modelUri);
			GridItem modelNode = newLine(grid, modelLabel, modelUri, modelContentType.id()); 
			setIcon(modelNode, 0, modelContentType.icon(modelUri));
			modelNode.setToolTipText(0, modelUri);
			
			if ( ctx.selection.contains( modelUri ) ) selectedNodes.add( modelNode );

			// Columns for formats
			int column1 = 1;
			for ( ContentType ct : contentToTypeMap.getValues(modelUri) ) {
				for ( Format format : typeToFormatMap.getValues(ct) ) {
					column1++;
					assertColumnIndex( column1 );
					modelNode.setText(column1, format.fileext());
					modelNode.setGrayed(column1, false);
					modelNode.setCheckable(column1, true);
					modelNode.setData( Integer.toString(column1), 
							new Content(modelUri, ct.id(), format.id(), modelLabel, format.fileext(), modelUri ) );
		   		    modelNode.setChecked(column1, hasInitialSelection(modelUri, format.id()));
		   		    modelNode.setToolTipText(column1, format.label());

					ImageDescriptor id = format.icon();
					if ( id!=null ) {
						Image icon = resourceManager.createImage(id);
						if ( icon!=null) modelNode.setImage(column1, icon);
					}
				}
			}
			
			for ( ContentType ct : content.getKeys() ) 
			{
				if ( ct.isModel() ) continue;
				// ContentType Node
			 	GridItem ctNode = newLine(modelNode, ct.label(), modelUri, ct.id());
			   	ctNode.setExpanded( true );
			   	ctNode.setBackground(0, contentTypeColor);
			   	ctNode.setBackground(contentTypeColor);
			   	setIcon(ctNode, 0, folderID);
			   	int contentCount = 0;
			   	ArrayList<Format> contentTypesFormats = new ArrayList<Format>();
			   	
			   	for ( String contentUri : content.getValues(ct) ) {
			   		// WORKAROUND: Should not be based on URI
			   		if ( !contentUri.startsWith(modelUri) ) continue;
			   		// Content Node
			   		String nodeLabel = labels.get(ct.id()).get(contentUri);
			   		GridItem contentNode = newLine(ctNode, nodeLabel, contentUri, ct.id());
			   		contentCount++;
			   		contentNode.setToolTipText(0, contentUri);
				   	setIcon(contentNode, 0, ct.icon(contentUri));
				   	
					if ( ctx.selection.contains( contentUri) ) selectedNodes.add( contentNode );
			    	
			   		int columnNumber = 0;
			   		
			   		// PDF Column
			   		List<Format> formats = typeToFormatMap.getValues(ct);
			   		if ( formats.contains( pdfFormat )) {
					   	if ( !contentTypesFormats.contains(pdfFormat) ) contentTypesFormats.add(pdfFormat);
			   			columnNumber++;
						assertColumnIndex( columnNumber );
			   			
			   			contentNode.setText(columnNumber, "  "+pdfFormat.fileext());
			   			contentNode.setGrayed(columnNumber, false);
			   		    contentNode.setCheckable(columnNumber, true);
			   		    contentNode.setChecked(columnNumber, hasInitialSelection(contentUri, pdfFormat.id()) );
			   		    contentNode.setToolTipText(columnNumber, pdfFormat.label());
					   	setIcon(contentNode, columnNumber, pdfFormat.icon());
					   						   	
					   	contentNode.setData(
					   			Integer.toString(columnNumber), 
					   			new Content(contentUri, ct.id(), pdfFormat.id(), nodeLabel, pdfFormat.fileext(), modelUri ) );
					   	
			   		} else {
					   	if ( !contentTypesFormats.contains(null) ) contentTypesFormats.add(null);
			   			columnNumber++;
						assertColumnIndex( columnNumber );
			   		}
			   		
			   		// Attachment Columns
			   		for (Format format : formats ) {
			   			if ( format==pdfFormat ) continue;
					   	if ( !contentTypesFormats.contains(format) ) contentTypesFormats.add(format);
			   			columnNumber++;
						assertColumnIndex( columnNumber );
			   			contentNode.setText(columnNumber, "  "+format.fileext());
			   			contentNode.setGrayed(columnNumber, false);
			   		    contentNode.setCheckable(columnNumber, true);
			   		    contentNode.setChecked(columnNumber, hasInitialSelection(contentUri, format.id()) );
			   		    contentNode.setToolTipText(columnNumber, format.label());					   	
			   		    setIcon(contentNode, columnNumber, format.icon());

					   	contentNode.setData(
					   			Integer.toString(columnNumber), 
					   			new Content(contentUri, ct.id(), format.id(), nodeLabel, format.fileext(), modelUri ) );
			   		}
			   	}
			   	
			   	// Add the *.pdf buttons
			   	int columnNumber = 0;
			   	Set<GridItem> gis = new HashSet<GridItem>();
			   	for ( Format format : contentTypesFormats ) {
			   		columnNumber++;
				   	ctNode.setBackground(columnNumber, contentTypeColor);
			   		if ( format == null ) continue;
			   		ctChecks.add( new CTCheck(ctNode, columnNumber, format) );
			   		ctNode.setCheckable(columnNumber, true);
			   		ctNode.setGrayed(columnNumber, true);
			   		//setIcon(ctNode, columnNumber, format.icon());
			   		gis.add(ctNode);
			   		//ctNode.setData(Integer.toString(columnNumber), format );
			   		//ctNode.setText(columnNumber, "*"+format.fileext());
			   	}
		   	
			   	if ( contentCount == 0 ) {
			   		ctNode.dispose();
			   	}
			   	
			}
		   	modelNode.setExpanded( true );
	    }
	    grid.setSelection( selectedNodes.toArray( new GridItem[selectedNodes.size()] ) );
	    if ( selectedNodes.size()>0 ) {
	    	GridItem first = selectedNodes.get(0);
	    	grid.setFocusItem( first );	    	
	    }

	    grid.addSelectionListener(ctChecksListener);
	    /*
	    grid.addSelectionListener( new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if ( e.item == null || e.item instanceof GridItem==false ) return;
				GridItem gi = (GridItem) e.item;
				GridColumn column = grid.getColumn( new Point(e.x, e.y) );
				if ( column == null ) return;
				int columnIndex = -1;
				int columnCount = grid.getColumnCount();
				for ( int i=0; i<columnCount; i++) {
					GridColumn gc = grid.getColumn(i);
					if ( gc == column ) {
						columnIndex = i;
						break;
					}
				}
				System.out.println(e.detail);
				System.out.println(e);
				System.out.println(columnIndex);
				String text = gi.getText(columnIndex);
				System.out.println(text);
			}
		});*/
	    
        setControl(grid);
        setPageComplete(true);
	}
	
	void setIcon(GridItem node, int index, ImageDescriptor icon) {
		if ( icon == null ) return;
		Image image = resourceManager.createImage(icon);
		if ( image == null ) return;
		node.setImage(index, image);
	}
 	
	/**
	 * Creates column index if doesn't exist.
	 * Column=2, is "Pages"
	 * Column>=3, is "Attachement"
	 * 
	 * @param columnIndex
	 */
	void assertColumnIndex(int columnIndex) {
		while ( columnIndex >= grid.getColumnCount() ) {
			int cc = grid.getColumnCount();
			
		    GridColumn column = new GridColumn(grid, SWT.CHECK);
		    column.setText( cc==1?"Pages":"Attachments");
		    column.setWidth( cc==1?150:200 );
		    
		    for (GridItem gi : grid.getItems()) {
				gi.setGrayed(cc, true);
				gi.setCheckable(cc, false);
		    }
			
		}	
	}
	
	boolean hasInitialSelection(String uri, String formatId) {
		for (Content c : initialSelection) {
			if ( !c.url.equals( uri ) ) continue;
			if ( c.formatId.equals("all") || c.formatId.equals(formatId) ) return true;
		}
		return false;
	}
	
	GridItem newLine(Object parent, String label, String url, String contentTypeId) {
		GridItem gi = null;
		if (parent instanceof Grid) {
			gi = new GridItem( (Grid)parent, SWT.NONE);
		} else {
			gi = new GridItem( (GridItem)parent, SWT.NONE);
		}
	 	
	  	gi.setText( label );
	  	if ( url!=null || contentTypeId!=null ) gi.setData( new Content(url, contentTypeId, null, label, null, null) );
	   	for ( int columnIndex=0; columnIndex<grid.getColumnCount(); columnIndex++ ) {
			gi.setGrayed(columnIndex, true);
			gi.setCheckable(columnIndex, false);
	   	}
	   		   	
	   	return gi;		
	}
		
	public void validatePage() {
		List<Content> newContentSelection = new ArrayList<Content>();				
		// Get list of content.. something must be checked
		int columnWidth = grid.getColumnCount();
    	Set<String> checkedFormats = new HashSet<String>();
	    for (GridItem gi : grid.getItems()) {
	    	/*
	    	checkedFormats.clear();
	    	GridItem parentItem = gi.getParentItem();
	    	if ( parentItem!=null ) {
		    	for (int c=0; c<columnWidth; c++) {
		    		if ( parentItem.getChecked(c) ) {
		    			Object data = parentItem.getData( Integer.toString(c) );
		    			if ( data==null || data instanceof Format == false ) continue;
		    			Format format = (Format) data;
		    			checkedFormats.add( format.id() );
		    		}
		    	}
	    	}*/
	    	
	    	for (int c=0; c<columnWidth; c++) {	    		
    			Object data = gi.getData( Integer.toString(c) );
    			if ( data==null || data instanceof Content == false ) continue;
	    		Content content = (Content) data;	    		
	    		if ( gi.getChecked(c) || checkedFormats.contains(content.formatId) ) {
	    			newContentSelection.add( content );
	    		}
	    	}
	    	
	    }
	    
	    contentSelection = newContentSelection;
	}
	
	public List<Content> getContentSelection() {
		return contentSelection;
	}

	public void savePrefs() {
		String str = ExportWizardResult.print( getContentSelection() );
		ctx.store.put(initialSelectionKey, str);
	}
	
	static class IndirectComparator implements Comparator<String> {
		Map<String, String> labels;

		@Override
		public int compare(String o1, String o2) {
			String l1 = null, l2 = null;
			if ( labels != null ) {
				l1 = labels.get(o1);
				l2 = labels.get(o2);
			} 
			if ( l1 == null ) l1 = o1;
			if ( l2 == null ) l2 = o2;
			return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(l1, l2);
		}
	}
	
	//// Code for following clicks on ContentType format specific checkboxes
	List<CTCheck> ctChecks = new ArrayList<CTCheck>();
	static class CTCheck {
		GridItem gi;
		int columnNumber;
		Format format;
		boolean lastKnownCheckedStatus = false;
		CTCheck(GridItem gi, int columnNumber, Format format) {
			this.gi = gi;
			this.columnNumber = columnNumber;
			this.format = format;
		}
		boolean previousSelection;
		boolean checkStatus() {
			return gi.getChecked(columnNumber);
		}
		void setCheck(boolean checked) {
			gi.setChecked(columnNumber, checked);
		}
	}
	
    SelectionListener ctChecksListener = new SelectionAdapter() {
		public void widgetSelected(SelectionEvent e) {
			if ( e.item == null || e.item instanceof GridItem==false ) return;
			int columnWidth = grid.getColumnCount();
			for ( CTCheck cc : ctChecks ) {
				boolean checked = cc.checkStatus();
				if ( checked == cc.lastKnownCheckedStatus ) continue;
				cc.lastKnownCheckedStatus = checked;
				
				for ( GridItem gi : cc.gi.getItems() ) {
					for ( int columnNumber = 0; columnNumber<columnWidth; columnNumber++ ) {
		    			Object data = gi.getData( Integer.toString( columnNumber ) );
		    			if ( data==null || data instanceof Content == false ) continue;
			    		Content content = (Content) data;	    		
						if ( !content.formatId.equals( cc.format.id() )) continue;
						gi.setChecked(columnNumber, checked);
					}
				}				
			}
			
		}
	};
	

	
}
