package org.simantics.export.core.intf;

import java.io.File;
import java.util.List;

import org.osgi.service.prefs.Preferences;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.type.RecordType;
import org.simantics.export.core.ExportContext;
import org.simantics.export.core.error.ExportException;
import org.simantics.export.core.manager.Content;

/**
 * Format class contains format related code.
 *
 * @author toni.kalajainen@semantum.fi
 */
public interface FormatClass {
	
	/**
	 * Create a new file. Return a format specific writer object. 
	 * 
	 * @param context contextual data
	 * @param outputFile
	 * @param options
	 * @return writer 
	 * @throws ExportException
	 */
	Object createFile( ExportContext context, File outputFile, Variant options ) throws ExportException;
	
	/**
	 * Open a file for reading.
	 * 
	 * @param context
	 * @param inputFile
	 * @param options
	 * @throws ExportException
	 */
	Object openFile( ExportContext context, File inputFile, Variant options ) throws ExportException;
	
	/**
	 * Close a reader or writer object.
	 * 
	 * ExportException is thrown if flushing of the file fails. This is a signal that
	 * the file is corrupted. The file is closed in all cases how ever. 
	 * 
	 * @param context
	 * @param handle
	 * @throws ExportException 
	 */
	void closeFile( ExportContext context, Object handle ) throws ExportException;

	/**
	 * Add attachments to an open file handle. This method applies only to 
	 * container files.  
	 * 
	 * @param context
	 * @param handle
	 * @param attachements
	 * @throws ExportException
	 */
	void addAttachment( ExportContext context, Object handle, List<Content> attachments ) throws ExportException;
	
	/**
	 * Get file format specific options.
	 * 
	 * @param context
	 * @return a record type describing options 
	 */
	RecordType options( ExportContext context ) throws ExportException;
	
	/**
	 * Validate that the options are usable for export.
	 * 
	 * @param context
	 * @param options
	 * @return a list of errors
	 * @throws ExportException
	 */
	List<String> validate( ExportContext context, Variant options ) throws ExportException;

	/**
	 * Fill options with default values.
	 * 
	 * @param options
	 * @throws ExportException 
	 */
	void fillDefaultPrefs( ExportContext context, Variant options ) throws ExportException;

	/**
	 * Save format options to preferences node(s).
	 * 
	 * A preference value may or may not exist in the options object.
	 * 
	 * @param options
	 * @param contentScopeNode
	 * @param workbenchScopeNode
	 * @throws ExportException
	 */
	void savePref( Variant options, Preferences contentScopeNode, Preferences workbenchScopeNode ) throws ExportException;
	
	/**
	 * Load format options from preferences node(s).
	 * 
	 * @param options
	 * @param contentScopeNode
	 * @param workbenchScopeNode
	 * @throws ExportException
	 */
	void loadPref( Variant options, Preferences contentScopeNode, Preferences workbenchScopeNode ) throws ExportException;
	
}
