package org.simantics.document.linking.wizard;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.ui.dialogs.ListDialog;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.PossibleObjectWithName;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.document.linking.report.evaluator.DBUtil;
import org.simantics.document.linking.report.templates.CustomizableContentProvider;
import org.simantics.document.linking.report.templates.ReportWriter;
import org.simantics.document.linking.report.templates.custom.CustomizableContent;
import org.simantics.document.linking.report.templates.custom.EvaluatorCustomizableContent;
import org.simantics.document.linking.report.templates.custom.SCLCustomizableContent;
import org.simantics.layer0.Layer0;
import org.simantics.utils.ui.validators.NotNullValidator;


/**
 * 
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class ReportCustomizationPage extends WizardPage{
	
	ReportWriter<?> reportWriter;
	Resource model;

	private Map<String,Composite> widgets = new LinkedHashMap<String, Composite>();
	private TabFolder tabFolder;
	private Map<String,TabItem> tabs = new LinkedHashMap<String, TabItem>();
	
	private Button loadButton;
	private Button saveButton;
	private Button resetButton;
	private Resource library;
	
	protected ReportCustomizationPage(String pageName) {
		super(pageName,pageName,null);
	}

	@Override
	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(1,false));
		setControl(composite);
		tabFolder = new TabFolder(composite, SWT.NONE);
		tabFolder.setLayout(new GridLayout());
		GridDataFactory.fillDefaults().align(SWT.FILL, SWT.FILL).grab(true, true).applyTo(tabFolder);
		Composite buttonComposite = new Composite(composite, SWT.NONE);
		GridDataFactory.fillDefaults().align(SWT.FILL, SWT.BOTTOM).grab(true, false).applyTo(buttonComposite);
		buttonComposite.setLayout(new GridLayout(3, true));
		loadButton = new Button(buttonComposite, SWT.PUSH);
		saveButton = new Button(buttonComposite, SWT.PUSH);
		resetButton = new Button(buttonComposite, SWT.PUSH);
		GridDataFactory.fillDefaults().align(SWT.FILL, SWT.CENTER).applyTo(loadButton);
		GridDataFactory.fillDefaults().align(SWT.FILL, SWT.CENTER).applyTo(saveButton);
		GridDataFactory.fillDefaults().align(SWT.FILL, SWT.CENTER).applyTo(resetButton);
		loadButton.setText("Load");
		saveButton.setText("Save");
		resetButton.setText("Defaults");
		
		loadButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				load();
			}
			
		});
		saveButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				save();
			};
		});
		resetButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				reset();
			};
		});
		updateContent();
	}
	
	public void setReportWriter(ReportWriter<?> reportWriter) {
		this.reportWriter = reportWriter;
		updateContentProviders();
		updateContent();
	}
	
	public void setModel(final Resource model) {
		this.model = model;
		
		
		try {
			// TODO : this code retrieves a proper location for storing the template configurations in a model. 
			//        this should be changed so that model resource somehow tells where to store the configurations.
			// Note: in other models, the location of templates is the model resource itself.
			this.library = Simantics.getSession().syncRequest(new Read<Resource>() {
				@Override
				public Resource perform(ReadGraph graph) throws DatabaseException {
					Layer0 l0 = Layer0.getInstance(graph);
					Resource lib =  graph.syncRequest(new PossibleObjectWithName(model, l0.ConsistsOf,"Library"));
					if (lib != null)
						return lib;
					
					return model;
				}
			});
		} catch (DatabaseException e) {
			this.library = model;
		}
	}
	
	private void updateContentProviders() {
		if (reportWriter instanceof CustomizableContentProvider) {
			LocalResourceManager manager = ((ReportWizard)getWizard()).manager;
			CustomizableContentProvider provider = (CustomizableContentProvider)reportWriter;
			Collection<String> ids = provider.getContentIds();
			for (String id : tabs.keySet()) {
				if (!ids.contains(id)) {
					tabs.remove(id).dispose();
					Composite c = widgets.get(id);
					if (c != null && !c.isDisposed())
						c.dispose();
				}
			}
			for (String id : ids) {
				CustomizableContent content = provider.getContent(id);
				if (content instanceof EvaluatorCustomizableContent) {
					TabItem item = tabs.get(id);
					if (item == null) {
						item = new TabItem(tabFolder, SWT.NONE);
						tabs.put(id, item);
					}
					item.setText(content.getCustomizationDescription());
					EvaluatorConfigurationWidget w = new EvaluatorConfigurationWidget(tabFolder,manager, SWT.NONE);
					w.setInput((EvaluatorCustomizableContent)content);
					GridDataFactory.fillDefaults().align(SWT.FILL, SWT.FILL).grab(true, true).applyTo(w);
					Composite c = widgets.put(id, w);
					if (c != null)
						c.dispose();
					item.setControl(w);
				} else if (content instanceof SCLCustomizableContent) {
					
				}
			}
		}
	}

	
	@Override
	public void setVisible(boolean visible) {
		super.setVisible(visible);
		updateContent();
	}
	
	
	private void updateContent() {
		for (Composite c : widgets.values()) {
			if (c instanceof EvaluatorConfigurationWidget)
				((EvaluatorConfigurationWidget)c).updateContent();
		}
		((Composite)getControl()).layout(true);
	}
	
	private void load() {
		try {
			List<NamedResource> templates = DBUtil.getTemplates(library);
			
			ListDialog dialog = new ListDialog(getShell());
			dialog.setContentProvider(new IStructuredContentProvider() {
				
				@Override
				public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
					
				}
				
				@Override
				public void dispose() {
					
				}
				
				@SuppressWarnings("unchecked")
				@Override
				public Object[] getElements(Object inputElement) {
					List<NamedResource> list = (List<NamedResource>)inputElement;
					return list.toArray();
				}
			});
			dialog.setLabelProvider(new LabelProvider());
			dialog.setInput(templates);
			if (dialog.open() != ListDialog.OK)
				return;
			Object result[] = dialog.getResult();
			if (result == null || result.length != 1)
				return;
			final NamedResource selectedTemplate = (NamedResource)result[0];
			Simantics.getSession().syncRequest(new ReadRequest() {
				
				@Override
				public void run(ReadGraph graph) throws DatabaseException {
					DBUtil.load(graph, selectedTemplate.getResource(), (CustomizableContentProvider)reportWriter);
					
				}
			});
			updateContentProviders();
			updateContent();
			
		} catch (DatabaseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
	}
	
	private void save() {
		try {
			
			InputDialog dialog = new InputDialog(getShell(), "Save report customization", "Give template name", "Name", new NotNullValidator("Name cannot be empty"));
			if (dialog.open() != InputDialog.OK)
				return;
			final String name = dialog.getValue();
			if (name == null || name.length() == 0)
				return;
			DBUtil.save(library, name, (CustomizableContentProvider)reportWriter);
		} catch (DatabaseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
			
	}
	
	private void reset() {
		if (reportWriter instanceof CustomizableContentProvider) {
			CustomizableContentProvider ccp = (CustomizableContentProvider)reportWriter;
			for (String id : ccp.getContentIds()) {
				ccp.setDefaultContent(id);
			}
			updateContentProviders();
			updateContent();
		}
	}
	
	

}
