/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - index based searching and graph manipulation (#4255)
 *******************************************************************************/
package org.simantics.debug.ui.internal;

import java.util.Set;

import org.eclipse.swt.widgets.Shell;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.debug.ui.GraphDebugger;
import org.simantics.debug.ui.ResourceSearch;
import org.simantics.debug.ui.SearchResourceDialog;
import org.simantics.layer0.Layer0;
import org.simantics.utils.Container;
import org.simantics.utils.DataContainer;

/**
 * @author Tuukka Lehtonen 
 */
public class DebugUtils {

    public static String getSafeLabel(ReadGraph graph, Resource r) throws DatabaseException {
        Layer0 l0 = Layer0.getInstance(graph);
        String name = NameUtils.getSafeName(graph, r);
        Statement stm = graph.getPossibleStatement(r, l0.HasLabel);
        if (stm != null) {
            String label = NameUtils.getSafeLabel(graph, r);
            if (!label.isEmpty() && !stm.isAsserted(r))
                name += " (" + label + ")";
        }
        return name;
    }

    public static String getSafeURI(ReadGraph graph, Resource r) throws DatabaseException {
        String uri = graph.getPossibleURI(r);
        if (uri != null)
            return uri;
        String name = NameUtils.getSafeName(graph, r, true);
        return name;
    }

    public static String getPossibleRootRelativePath(ReadGraph graph, Resource r) throws DatabaseException {
        Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(r));
        String uri = graph.getPossibleURI(r);
        if (indexRoot != null && uri != null) {
            Layer0 L0 = Layer0.getInstance(graph);
            Set<Resource> types = graph.getTypes(indexRoot);
            if (!types.contains(L0.Ontology)) {
                Resource indexRootParent = graph.getPossibleObject(indexRoot, L0.PartOf);
                if (indexRootParent != null) {
                    String rootParentUri = graph.getPossibleURI(indexRootParent);
                    if (rootParentUri != null) {
                        return URIStringUtils.unescape( uri.substring(rootParentUri.length()+1) );
                    }
                }
            }
        }
        return uri;
    }

    @SuppressWarnings("unchecked")
    public static void addResource(Session s, GraphDebugger debugger) throws DatabaseException {
        Shell shell = debugger.getShell();

        SearchResourceDialog rld = new SearchResourceDialog(s, false, shell, "Create New Resource");
        rld.setBlockOnOpen(true);
        rld.setResourceFilter(ResourceSearch.FILTER_TYPES);

        Resource subject_ = debugger.getDebuggerLocation();
        if (subject_ == null) {
            rld.setBlockOnOpen(true);
            rld.setMessage("Select Subject");
            rld.setInitialSelections(new Object[] {});
            if (rld.open()!=org.eclipse.jface.window.Window.OK) return;
            if (rld.getResult()==null) return;
            subject_ = ((Container<Resource>)rld.getResult()[0]).get();
        }
        final Resource subject = subject_;

        rld.setBlockOnOpen(true);
        rld.setResourceFilter(ResourceSearch.FILTER_RELATIONS);
        rld.setMessage("Select Predicate");
        rld.setInitialSelections(new Object[] {});
        if (rld.open()!=org.eclipse.jface.window.Window.OK) return;
        if (rld.getResult()==null) return;
        final Resource predicate = ((Container<Resource>)rld.getResult()[0]).get();

        rld.setMessage("Select Type of New Object Instance");
        rld.setResourceFilter(ResourceSearch.FILTER_TYPES);
        rld.setInitialSelections(new Object[] {});
        if (rld.open()!=org.eclipse.jface.window.Window.OK) return;
        if (rld.getResult()==null) return;
        final Resource type = ((Container<Resource>)rld.getResult()[0]).get();
        final DataContainer<Resource> result = new DataContainer<Resource>();
        s.syncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph g) throws DatabaseException {
                Resource r = g.newResource();
                g.claim(r, Layer0.getInstance(g).InstanceOf, type);
                g.claim(subject, predicate, r);
                result.set(r);
            }
        });

        if (result.get()!=null)
            debugger.changeLocation(result.get());
    }

    @SuppressWarnings("unchecked")
    public static void addStatement(Session s, GraphDebugger debugger) throws DatabaseException {
        Shell shell = debugger.getShell();
        SearchResourceDialog rld = new SearchResourceDialog(s, false, shell, "Create New Statement");

        Resource subject_ = debugger.getDebuggerLocation();
        if (subject_ == null) {
            rld.setBlockOnOpen(true);
            rld.setMessage("Select Subject");
            rld.setInitialSelections(new Object[] {});
            if (rld.open()!=org.eclipse.jface.window.Window.OK) return;
            if (rld.getResult()==null) return;
            subject_ = ((Container<Resource>)rld.getResult()[0]).get();
        }
        final Resource subject = subject_;

        rld.setBlockOnOpen(true);
        rld.setResourceFilter(ResourceSearch.FILTER_RELATIONS);
        rld.setMessage("Select Predicate");
        rld.setInitialSelections(new Object[] {});
        if (rld.open()!=org.eclipse.jface.window.Window.OK) return;
        if (rld.getResult()==null) return;
        final Resource predicate = ((Container<Resource>)rld.getResult()[0]).get();

        rld.setResourceFilter(ResourceSearch.FILTER_ALL);
        rld.setMessage("Select Object");
        rld.setInitialSelections(new Object[] {});
        if (rld.open()!=org.eclipse.jface.window.Window.OK) return;
        if (rld.getResult()==null) return;
        final Resource object = ((Container<Resource>)rld.getResult()[0]).get();

        s.syncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph g) throws DatabaseException {
                g.claim(subject, predicate, object);
            }
        });
    }

    public static void find(Session s, GraphDebugger debugger) {
        Shell shell = debugger.getShell();
        SearchResourceDialog rld = new SearchResourceDialog(s, false, shell, "Select Resource to View");
        rld.setBlockOnOpen(true);
        if (rld.open()!=org.eclipse.jface.window.Window.OK) return;
        if (rld.getResult()==null) return;
        for (Object o : rld.getResult()) {
            @SuppressWarnings("unchecked")
            Container<Resource> rc = (Container<Resource>) o;
            debugger.changeLocation(rc.get());
        }
    }

}
