/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.procore.cluster;

import gnu.trove.map.hash.TObjectIntHashMap;

import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ClusterI.Procedure;
import org.simantics.db.impl.ClusterSupport;
import org.simantics.db.impl.Modifier;
import org.simantics.db.impl.Table;
import org.simantics.db.impl.TableFactory;
import org.simantics.db.impl.TableSizeListener;
import org.simantics.db.service.ClusterUID;
import org.simantics.db.service.ResourceUID;

final class ForeignElement {
    private static final int CLUSTER_OFFSET = 0; // resourceUID
    private static final int SIZE_OF = 3;

    static void constructForeign(long[] table, int index, ResourceUID resourceUID) {
        int i = CLUSTER_OFFSET + index;
        resourceUID.toLong(table, i);
    }
    static void destructForeign(long[] table, int index) {
        int i = CLUSTER_OFFSET + index;
        table[i++] = 0;
        table[i++] = 0;
        table[i++] = 0;
    }
    static int getSizeOf() {
        return SIZE_OF;
    }
    static ResourceUID getResourceUID(long[] table, int index) {
        int i = CLUSTER_OFFSET + index;
        return new ResourceUID(table, i);
    }
    static void fillResourceUID(long[] table, int index, ClusterSmall cluster) {
        int i = CLUSTER_OFFSET + index;
        cluster.clusterUID1 = table[i];
        cluster.clusterUID2 = table[i+1];
        cluster.executeIndex = (short)table[i+2];
    }
}

public final class ForeignTable extends Table<long[]>
{
    public ForeignTable(TableSizeListener sizeListener, int[] header, int headerBase) {
        super(TableFactory.getLongFactory(), sizeListener, header, headerBase);
    }
    public ForeignTable(TableSizeListener sizeListener, int[] header, int headerBase, long[] longs) {
        super(TableFactory.getLongFactory(), sizeListener, header, headerBase, longs);
    }
    public int getUsedSize() {
        return getTableCount();
    }
    TObjectIntHashMap<ClusterUID> getHashMap()
    throws DatabaseException {
        int ELEMENT_SIZE = ForeignElement.getSizeOf();
        final int TABLE_SIZE = getTableCount();
        final int FOREIGN_COUNT = getForeignCount();
        final int BASE = this.getTableBase();
        final int TOP = BASE + TABLE_SIZE * ELEMENT_SIZE;
        long[] table = this.getTable();
        int count = 0;
        int index = ZERO_SHIFT;
        TObjectIntHashMap<ClusterUID> hm = new TObjectIntHashMap<ClusterUID>(FOREIGN_COUNT);
        for (int i=BASE; i<TOP && count<FOREIGN_COUNT;
        i+=ELEMENT_SIZE, ++index) {
            if (0 == table[i] && 0 == table[i+1] && 0 == table[i+2])
                continue; // element has been deleted
            hm.put(ClusterUID.make(table[i], table[i+1]), index);
            ++count;
        }
        if (FOREIGN_COUNT != hm.size())
            throw new DatabaseException("Foreign table has been corrupted.");
        return hm;
    }
    TObjectIntHashMap<ResourceUID> getResourceHashMap()
    throws DatabaseException {
        int ELEMENT_SIZE = ForeignElement.getSizeOf();
        assert(ELEMENT_SIZE == 3);
        final int TABLE_SIZE = getTableCount();
        final int FOREIGN_COUNT = getForeignCount();
        final int BASE = this.getTableBase();
        final int TOP = BASE + TABLE_SIZE * ELEMENT_SIZE;
        long[] table = this.getTable();
        int count = 0;
        int index = ZERO_SHIFT;
        TObjectIntHashMap<ResourceUID> hm = new TObjectIntHashMap<ResourceUID>(FOREIGN_COUNT);
        for (int i=BASE; i<TOP && count<FOREIGN_COUNT;
        i+=ELEMENT_SIZE, ++index) {
            if (0 == table[i] && 0 == table[i+1] && 0 == table[i+2])
                continue; // element has been deleted
            hm.put(new ResourceUID(table[i], table[i+1], table[i+2]), index);
            ++count;
        }
        if (FOREIGN_COUNT != hm.size())
            throw new DatabaseException("Foreign table has been corrupted. count=" + FOREIGN_COUNT + "size=" + hm.size());
        return hm;
    }
    int createForeign(ResourceUID uid) {
        int index = getTableCount();
        int size = ForeignElement.getSizeOf();
        int foreignIndex = createNewElement(size);
        int realIndex = checkIndexAndGetRealIndex(foreignIndex, size);
        ForeignElement.constructForeign(getTable(), realIndex, uid);
        incForeignCount();
        return index + ZERO_SHIFT;
    }
    void deleteForeign(int foreignIndex) {
        int realIndex = checkIndexAndGetRealIndex(foreignIndex);
        ForeignElement.destructForeign(getTable(), realIndex);
        decForeignCount();
    }
    public final ResourceUID getResourceUID(int foreignIndex) {
        return ForeignElement.getResourceUID(table, checkIndexAndGetRealIndex(foreignIndex)); 
    }
    int getForeignCount() {
        return getExtra(FOREIGN_COUNT_INDEX);
    }
    private static final int FOREIGN_COUNT_INDEX = 0;
    private int incForeignCount() {
        int count = getExtra(FOREIGN_COUNT_INDEX) + 1;
        setExtra(FOREIGN_COUNT_INDEX, count);
        return count;
    }
    private int decForeignCount() {
        int count = getExtra(FOREIGN_COUNT_INDEX) - 1;
        setExtra(FOREIGN_COUNT_INDEX, count);
        return count;
    }
    private int checkIndexAndGetRealIndex(int foreignIndex) {
        int index = (foreignIndex - ZERO_SHIFT) * ForeignElement.getSizeOf() + ZERO_SHIFT;
        int realIndex = checkIndexAndGetRealIndex(index, ForeignElement.getSizeOf());
        return realIndex;
    }
    @Override
    public <Context> boolean foreach(int setIndex, Procedure procedure, Context context,
            ClusterSupport support, Modifier modifier) throws DatabaseException {
        throw new UnsupportedOperationException();
    }
}
