package fi.vtt.simantics.procore.internal;

import org.simantics.db.Resource;
import org.simantics.db.ResourceSerializer;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.InvalidResourceReferenceException;
import org.simantics.db.impl.ClusterI;
import org.simantics.db.impl.ClusterI.PredicateProcedure;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.procore.cluster.ClusterImpl;
import org.simantics.db.procore.cluster.ClusterTraits;
import org.simantics.db.service.ResourceUID;
import org.simantics.db.service.SerialisationSupport;

import fi.vtt.simantics.procore.internal.SessionImplSocket.ResourceSerializerImpl;

public class SerialisationSupportImpl implements SerialisationSupport {

	final private SessionImplSocket session;
	final private ResourceSerializerImpl serializer;
	
	SerialisationSupportImpl(SessionImplSocket session) {
		this.session = session;
		this.serializer = session.resourceSerializer;
	}
	
    @Override
    public ResourceSerializer getResourceSerializer() {
        return serializer;
    }

	@Override
	public long getRandomAccessId(int id) {
		try {
			return serializer.createRandomAccessId(id);
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		} catch (Throwable t) {
			Logger.defaultLogError(t);
		}
		return 0;
	}
	
	@Override
	public int getTransientId(Resource resource) throws DatabaseException {
		return serializer.getTransientId(resource);
	}
	
	@Override
	public Resource getResource(long randomAccessId) throws DatabaseException {
		return serializer.getResource(randomAccessId);
	}
	
	@Override
	public int getTransientId(long randomAccessId) throws DatabaseException {
		return serializer.getTransientId(randomAccessId);
	}

	@Override
	public long getRandomAccessId(Resource resource) throws DatabaseException {
		return serializer.getRandomAccessId(resource);
	}

	@Override
	public Resource getResource(int transientId) throws DatabaseException {
		return session.getResource(transientId);
	}

	@Override
	public ResourceUID getUID(Resource resource) throws DatabaseException {
		ResourceImpl impl = (ResourceImpl)resource;
		int resourceKey = impl.id;
		ClusterImpl cluster = session.clusterTable.getClusterByResourceKey(resourceKey);
		return cluster.clusterUID.toRID(ClusterTraits.getResourceIndexFromResourceKey(resourceKey));
	}

	static class ExistsPredicateProcedure implements PredicateProcedure<Integer> {
		
		boolean exists = false;

		@Override
		public boolean execute(Integer c, final int predicateKey, int objectIndex) {
			exists = true;
			return false;
		}

	}
	
	@Override
	public Resource getResource(ResourceUID uid) throws DatabaseException {
        ClusterI cluster = session.clusterTable.getClusterByClusterUIDOrMakeProxy(uid.asCID());
        int key = ClusterTraits.createResourceKey(cluster.getClusterKey(), (int) uid.getIndex());
        if (cluster.hasResource(key, session.clusterTranslator)) {
        	ExistsPredicateProcedure pp = new ExistsPredicateProcedure();
			cluster.forPredicates(key, pp, 0, session.clusterTranslator);
			if(pp.exists) {
				return new ResourceImpl(session.resourceSupport, key);
			} else if (cluster.hasValue(key, session.clusterTranslator)) {
				return new ResourceImpl(session.resourceSupport, key);
			}
        }
        throw new InvalidResourceReferenceException("Resource with uid = " + uid + " does not exist.");
	}
    
}
