/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import gnu.trove.map.hash.TObjectIntHashMap;

import java.util.concurrent.Semaphore;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.common.WriteBindings;
import org.simantics.db.common.exception.DebugException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.procedure.ListenerBase;

final public class NamespaceIndex extends StringQuery<InternalProcedure<TObjectIntHashMap<String>>> {
    
//	public ArrayList<InternalProcedure<TObjectIntHashMap<String>>> procs = null;
	
    private NamespaceIndex(final String id) {
        super(id);
    }
    
    final static void runner(ReadGraphImpl graph, final String id, final QueryProcessor provider, NamespaceIndex cached, final CacheEntry parent, final ListenerBase listener, final InternalProcedure<TObjectIntHashMap<String>> procedure) {

    	NamespaceIndex entry = cached != null ? cached : (NamespaceIndex)provider.namespaceIndexMap22.get(id); 
        if(entry == null) {
        	
        	entry = new NamespaceIndex(id);
        	entry.setPending();
        	entry.clearResult(provider.querySupport);
        	entry.putEntry(provider);

            provider.performForEach(graph, entry, parent, listener, procedure);
            
        } else {

            if(entry.isPending()) {
            	synchronized(entry) {
                    if(entry.isPending()) {
                        throw new IllegalStateException();
//                    	if(entry.procs == null) entry.procs = new ArrayList<InternalProcedure<TObjectIntHashMap<String>>>(); 
//                    	entry.procs.add(procedure);
//                        provider.registerDependencies(graph, entry, parent, listener, procedure, false);
//                    	return;
                    }
            	}
            }
            provider.performForEach(graph, entry, parent, listener, procedure);
        }
        
    }
    
    final public static void queryEach(ReadGraphImpl graph, final String id, final QueryProcessor provider, final CacheEntry parent, final ListenerBase listener, final InternalProcedure<TObjectIntHashMap<String>> procedure) {

    	final NamespaceIndex entry = (NamespaceIndex)provider.namespaceIndexMap22.get(id);
        
    	if(parent == null && listener == null && entry != null && entry.isReady()) {
    		entry.performFromCache(graph, provider, procedure);
    		return;
    	}
    	
        runner(graph, id, provider, entry, parent, listener, procedure);
        
    }
     
 	@Override
 	public NamespaceIndex getEntry(QueryProcessor provider) {
        return provider.namespaceIndexMap22.get(id);
 	}
 	
 	@Override
 	public void putEntry(QueryProcessor provider) {
        provider.namespaceIndexMap22.put(id, this);
 	}

 	@Override
 	final public void removeEntry(QueryProcessor provider) {
 		provider.namespaceIndexMap22.remove(id);
 	}
 	
	final private void index(ReadGraphImpl graph, final QueryProcessor provider, int root, final InternalProcedure<TObjectIntHashMap<String>> procedure) {
		
		if(root == 0) {
            add2(graph, null);
            procedure.execute(graph, null);
//            System.err.println("NamespaceIndex[" + id + "]->null");
            return;
		}

		final int consistsOf = provider.getConsistsOf();
		final int hasName = provider.getHasName();
        
        final TObjectIntHashMap<String> result = new TObjectIntHashMap<String>();
        
        Objects.runner(graph, root, consistsOf, graph.parent, null, new SyncIntProcedure() {
        	
        	@Override
			public void run(ReadGraphImpl graph) {
        		
        		if(isPending()) { 
                    add2(graph, result);
                    procedure.execute(graph, result);
//                    System.err.println("NamespaceIndex[" + id + "]->" + result.size());
        		} else {
        			procedure.exception(graph, (Throwable)statusOrException);
        		}
				
			}

			@Override
			public void finished(ReadGraphImpl graph) {
			    
				dec(graph);
				
			}

        	@Override
        	public void execute(ReadGraphImpl graph, final int obj) {
        		
        		//System.out.println(id + " => " + obj);

        		inc();
        		
        		Objects.runner(graph, obj, hasName, graph.parent, null, new IntProcedure() {
                	
                	@Override
                	public void execute(ReadGraphImpl graph, int i) {

                		inc();

                		ValueQuery.queryEach(graph, i, NamespaceIndex.this, null, new InternalProcedure<byte[]>() {
                			
                			@Override
                			public void execute(ReadGraphImpl graph, byte[] value) {
                				
                				if(value != null) {

	                                try {

	                                	Binding b = WriteBindings.STRING;
	                                    Serializer serializer = b.serializer();
	                                    final String part = (String)serializer.deserialize(value);
	
	                                    synchronized(result) {
	                                        Object previous = result.put(URIStringUtils.escape(part), obj);
	                                        // TODO: this is not the most elegant solution
	                                        if(previous != null) previous = "";
	                                    }
	                                    
	                	        	} catch (Throwable e) {
	                                    if(DebugException.DEBUG) new DebugException(e).printStackTrace();
	                                }
	                	        	
                				}
                				
                	        	dec(graph);
                	        	
                			}
                			
                			@Override
                			public void exception(ReadGraphImpl graph, Throwable t) {
        	 					except(t);
                				dec(graph);
                            }

                		});
                		
                	}
                	
                	@Override
                	public void finished(ReadGraphImpl graph) {
                		dec(graph);
                	}
        			
        			@Override
        			public void exception(ReadGraphImpl graph, Throwable t) {
	 					except(t);
                        dec(graph);
                    }

                });

        	}
        	
        });
        
    }

    @Override
 	public void computeForEach(ReadGraphImpl graph, final QueryProcessor processor, final InternalProcedure<TObjectIntHashMap<String>> procedure) {
    	
//    	System.err.println("NamespaceIndex " + id);
    	
        if("http://".equals(id) || "http:/".equals(id)) {
            index(graph, processor, processor.getRootLibrary(), procedure);
        } else {
            final String[] parts = URIStringUtils.splitURI(id);
            if(parts != null) {
                NamespaceIndex.queryEach(graph, parts[0], processor, this, null, new InternalProcedure<TObjectIntHashMap<String>>() {
    
                    @Override
                    public void execute(ReadGraphImpl graph, TObjectIntHashMap<String> index) {
    
                        if(index != null) {
                            index(graph, processor, index.get(parts[1]), procedure);
                        } else {
                            add2(graph, null);
                            procedure.execute(graph, null);
//                            System.err.println("NamespaceIndex[" + id + "]->null");
                        }
                        
                    }
    
                    @Override
                    public void exception(ReadGraphImpl graph, Throwable t) {
                        if(DebugException.DEBUG) new DebugException(t).printStackTrace();
                        except(t);
                        procedure.exception(graph, t);
                    }
    
                });
            } else {
                add2(graph, null);
                procedure.execute(graph, null);
//                System.err.println("NamespaceIndex[" + id + "]->null");
            }

        }
        
    }

    @Override
    public String toString() {
    	return "NamespaceIndex[" + id + "]";
    }

    synchronized private void add(TObjectIntHashMap<String> result) {
    	
    	throw new Error("Not possible!");
    	
    }

    private void add2(ReadGraphImpl graph, TObjectIntHashMap<String> result) {
    	
    	if(!isPending()) {
        	new Exception(""+hashCode()).printStackTrace();
    	}
    	
    	assert(isPending());

//        ArrayList<InternalProcedure<TObjectIntHashMap<String>>> p = null;

        synchronized(this) {

            setResult(result);
        	setReady();
//            p = procs;
//            procs = null; 
        
        }
        
//        if(p != null) {
//        
//	        for(InternalProcedure<TObjectIntHashMap<String>> proc : p) proc.execute(graph, result);
//	        
//        }
    	
    }
    
    @Override
    public void performFromCache(ReadGraphImpl graph, QueryProcessor provider, InternalProcedure<TObjectIntHashMap<String>> procedure) {
        
        assert(isReady());
        
    	if(handleException(graph, procedure)) return;
        
        procedure.execute(graph, (TObjectIntHashMap<String>)getResult());
        
    }
    
    @Override
    public synchronized void recompute(ReadGraphImpl graph, QueryProcessor provider) {
        
        final Semaphore s = new Semaphore(0);
        
        computeForEach(graph, provider, new InternalProcedure<TObjectIntHashMap<String>>() {

            @Override
            public void execute(ReadGraphImpl graph, TObjectIntHashMap<String> result) {
                s.release();
            }
			
			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
                if(DebugException.DEBUG) new DebugException(t).printStackTrace();
				throw new Error("Error in recompute.", t);
            }

        });
        
        while(!s.tryAcquire()) {
        	provider.resume(graph);
        }
        
    }
    
}

