/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.processor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Semaphore;
import java.util.function.Consumer;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.AsyncRequestProcessor;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.procedure.adapter.AsyncMultiProcedureAdapter;
import org.simantics.db.common.procedure.adapter.ProcedureAdapter;
import org.simantics.db.common.procedure.wrapper.NoneToAsyncProcedure;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.CancelTransactionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncListener;
import org.simantics.db.procedure.AsyncMultiListener;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.Listener;
import org.simantics.db.procedure.MultiListener;
import org.simantics.db.procedure.MultiProcedure;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.procedure.SyncListener;
import org.simantics.db.procedure.SyncMultiListener;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.procedure.SyncProcedure;
import org.simantics.db.request.AsyncMultiRead;
import org.simantics.db.request.AsyncRead;
import org.simantics.db.request.DelayedWrite;
import org.simantics.db.request.DelayedWriteResult;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.Read;
import org.simantics.db.request.ReadInterface;
import org.simantics.db.request.Write;
import org.simantics.db.request.WriteInterface;
import org.simantics.db.request.WriteOnly;
import org.simantics.db.request.WriteOnlyResult;
import org.simantics.db.request.WriteResult;
import org.simantics.utils.DataContainer;
import org.simantics.utils.datastructures.Pair;

public class MergingGraphRequestProcessor implements RequestProcessor {

    private static class SyncWriteRequestAdapter implements Write {

        private Semaphore semaphore = new Semaphore(0);
        private Object request;
        private Throwable exception;
        SyncWriteRequestAdapter(Write r) {
            this.request = r;
        }
        SyncWriteRequestAdapter(WriteOnly r) {
            this.request = r;
        }
//        @Override
//        public GraphRequestStatus perform(Graph g) throws Exception {
//            return perform((ReadGraph)g);
//        }
        @Override
        public void perform(WriteGraph g) throws DatabaseException, CancelTransactionException {
            if(request instanceof Write) {
                ((Write)request).perform(g);
            } else if(request instanceof DelayedWrite) {
                    ((DelayedWrite)request).perform(g);
            } else {
                ((WriteOnly)request).perform(g);
            }
        }
//        @Override
//        public String getId() {
//            if(request instanceof WriteGraphRequest) {
//                return ((WriteGraphRequest)request).getId();
//            } else {
//                return null;
//            }
//        }
//        @Override
//        public void requestCompleted(GraphRequestStatus status) {
//            if(request instanceof WriteGraphRequest) {
//                ((WriteGraphRequest)request).requestCompleted(status);
//            } else {
//            }
//        }
//        @Override
//        public void handleException(Throwable e) {
//            this.exception = e;
//            if(request instanceof WriteGraphRequest) {
//                ((WriteGraphRequest)request).handleException(e);
//            }
//        }

        public void throwOrWrapException() {
            if (exception == null)
                return;
            if (exception instanceof RuntimeException)
                throw (RuntimeException) exception;
            if (exception instanceof Error)
                throw (Error) exception;
            throw new RuntimeException("See cause for the real exception.", exception);
        }

        public void acquire() {
            try {
                semaphore.acquire();
            } catch (InterruptedException e) {
	    		Logger.defaultLogError(e);
            }
        }

        public void release() {
            semaphore.release();
        }

        @Override
        public String toString() {
            return "SyncWriteRequestAdapter " + request;
        }

    }

    long transactionKeepalivePeriod;

    /**
     * Synchronization object for implementing {@link #synchronize()}.
     * {@link Object#notifyAll()} is invoked for this lock object every time a
     * single transaction is completed, thereby releasing all waiters in
     * {@link #synchronize()}.
     */
    Object barrier = new Object();

    Set<Pair<Object, Object>> requestSet = new HashSet<Pair<Object, Object>>();
    LinkedList<Pair<Object, Object>> requestQueue = new LinkedList<Pair<Object, Object>>();
    boolean hasAlreadyRequest = false;

    /**
     * A set of requests which {@link #synchronize()} is depending on at the
     * moment. Every time a request within this set is completed, some thread in
     * {@link #synchronize()} should be released.
     */
//    Set<Object> barrierRequests = new HashSet<Object>();
    Set<Object> syncRequests = new HashSet<Object>();

    private String name;

    private AsyncRequestProcessor processor;

    public MergingGraphRequestProcessor(String name, AsyncRequestProcessor processor, long transactionKeepalivePeriod) {
        this.name = name;
        this.processor = processor;
        this.transactionKeepalivePeriod = transactionKeepalivePeriod;
    }

    public MergingGraphRequestProcessor(AsyncRequestProcessor processor, long transactionKeepalivePeriod) {
        this.name = "MergingGraphRequestProcessor" + UUID.randomUUID().toString();
        this.processor = processor;
        this.transactionKeepalivePeriod = transactionKeepalivePeriod;
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    protected class MergedRead extends ReadRequest {

        Pair<Object, Object> currentRequest;

//        RunnerReadGraphRequest(GraphRequestProcessor processor) {
//            super(processor);
//        }
//
//        @Override
//        public void completed(boolean value) {
////        	System.out.println(this + "MGRP read completed");
////            synchronized (MergingGraphRequestProcessor.this) {
////                if (requestQueue.isEmpty())
////                    hasAlreadyRequest = false;
////                else
////                    newTransaction();
////            }
//        }

        @Override
        public void run(ReadGraph graph) {

//            System.out.println(MergingGraphRequestProcessor.this + " reads");

            while (true) {

                synchronized (MergingGraphRequestProcessor.this) {

                    // Release #synchronize() invokers if necessary.
//                    if (currentRequest != null && barrierRequests.contains(currentRequest)) {
//                        synchronized (barrier) {
//                            barrier.notifyAll();
//                        }
//                    }

                    if(requestQueue.isEmpty()) {
                        if (transactionKeepalivePeriod > 0) {
//                            System.out.println("MGRP [" + MergingGraphRequestProcessor.this + "] waits " + transactionKeepalivePeriod + " ms. in " + Thread.currentThread() );
                            try {
                                MergingGraphRequestProcessor.this.wait(transactionKeepalivePeriod);
                            } catch (InterruptedException e) {
                	    		Logger.defaultLogError(e);
                            }
                            if (requestQueue.isEmpty())
                                break;
                        } else
                            break;
                    }

                    Object nextRequest = requestQueue.peekFirst().first;
                    if(nextRequest instanceof Write || nextRequest instanceof DelayedWrite) {
                        break;
                    }

                    currentRequest = requestQueue.remove(0);
                    requestSet.remove(currentRequest);

                }

//                ReadGraphRequest req = (ReadGraphRequest)currentRequest.first;

                if( syncRequests.contains(currentRequest.first)) {

                    try {

                        if(currentRequest.second instanceof AsyncProcedure<?>) {
                            if(currentRequest.first instanceof Read) {
                                Read req = (Read)currentRequest.first;
                                graph.syncRequest(req, (AsyncProcedure<?>)currentRequest.second);
                            } else {
                                AsyncRead req = (AsyncRead)currentRequest.first;
                                graph.syncRequest(req, (AsyncProcedure<?>)currentRequest.second);
                            }
                        } else {
                            AsyncMultiRead req = (AsyncMultiRead)currentRequest.first;
                            graph.syncRequest(req, (AsyncMultiProcedure<?>)currentRequest.second);
                        }

                    } catch(Throwable t) {

        	    		Logger.defaultLogError(t);

                        if(currentRequest.second instanceof AsyncProcedure<?>) {
                            ((AsyncProcedure<?>)currentRequest.second).exception(graph, t);
                        } else {
                            ((AsyncMultiProcedure<?>)currentRequest.second).exception(graph, t);
                        }

                    }

                    synchronized (currentRequest.first) {
                        syncRequests.remove(currentRequest.first);
//                    	System.out.println("notifying " + currentRequest.first);
                        currentRequest.first.notify();
                    }


                } else {

                    try{

                        if(currentRequest.second instanceof AsyncProcedure<?>) {
                            if(currentRequest.first instanceof AsyncRead) {
                                AsyncRead req = (AsyncRead)currentRequest.first;
                                graph.asyncRequest(req, (AsyncProcedure<?>)currentRequest.second);
                            } else {
                                Read req = (Read)currentRequest.first;
                                graph.asyncRequest(req, (AsyncProcedure<?>)currentRequest.second);
                            }
                        } else {
                            AsyncMultiRead req = (AsyncMultiRead)currentRequest.first;
                            graph.asyncRequest(req, (AsyncMultiProcedure<?>)currentRequest.second);
                        }

                    } catch(Throwable t) {

        	    		Logger.defaultLogError(t);

                        if(currentRequest.second instanceof AsyncProcedure<?>) {
                            ((AsyncProcedure<?>)currentRequest.second).exception(graph, t);
                        } else {
                            ((AsyncMultiProcedure<?>)currentRequest.second).exception(graph, t);
                        }

                    }
                }

            }

//            System.out.println(MergingGraphRequestProcessor.this + " read completed");

            synchronized (MergingGraphRequestProcessor.this) {
                if (requestQueue.isEmpty())
                    hasAlreadyRequest = false;
                else
                    newTransaction();
            }

        }

        @Override
        public String toString() {
            return "MergedRead[" + requestQueue.size() + " requests]";
        }

    }

    protected class RunnerWriteGraphRequest extends WriteRequest {

        Pair<Object, Object> currentRequest;
        HashMap<String, String> metadata = new HashMap<String, String>();

        @Override
        public void perform(WriteGraph graph) throws DatabaseException {

//            System.out.println(MergingGraphRequestProcessor.this + " writes");

            while (true) {

                synchronized (MergingGraphRequestProcessor.this) {

                    // Release #synchronize() invokers if necessary.
//                    if (currentRequest != null && barrierRequests.contains(currentRequest)) {
//                        synchronized (barrier) {
//                            barrier.notifyAll();
//                        }
//                    }

                    if(requestQueue.isEmpty()) {
                        if (transactionKeepalivePeriod > 0) {
                            try {
                                MergingGraphRequestProcessor.this.wait(transactionKeepalivePeriod);
                            } catch (InterruptedException e) {
                	    		Logger.defaultLogError(e);
                            }
                            if (requestQueue.isEmpty())
                                break;
                        } else
                            break;
                    }

                    Object nextRequest = requestQueue.peekFirst().first;
                    if(nextRequest instanceof AsyncMultiRead || nextRequest instanceof AsyncRead || nextRequest instanceof Read) {
                        break;
                    }

                    currentRequest = requestQueue.remove(0);
                    requestSet.remove(currentRequest);

                }

                @SuppressWarnings("unchecked")
                Consumer<Throwable> callback = (Consumer<Throwable>)currentRequest.second;

                if (currentRequest.first.getClass().equals(SyncWriteRequestAdapter.class)) {

                    SyncWriteRequestAdapter adapter = (SyncWriteRequestAdapter)currentRequest.first;

                    try {
//                        System.out.println("merg.sync " + adapter);
                        graph.syncRequest(adapter);
                        if(callback != null) callback.accept(null);
                    } catch(Throwable t) {
        	    		Logger.defaultLogError(t);
                        if(callback != null) callback.accept(t);
                    }

                    adapter.release();
//                    System.out.println("merg.sync.release " + adapter);

                } else {

                    try {
                        if(currentRequest.first instanceof Write) graph.syncRequest((Write)currentRequest.first); 
                        else if(currentRequest.first instanceof DelayedWrite) graph.syncRequest((DelayedWrite)currentRequest.first); 
                        if(callback != null) callback.accept(null);
                    } catch(Throwable t) {
        	    		Logger.defaultLogError(t);
                        if(callback != null) callback.accept(t);
                    }

                }

            }

//            System.out.println(MergingGraphRequestProcessor.this + " write completed");

            synchronized (MergingGraphRequestProcessor.this) {
                if (requestQueue.isEmpty())
                    hasAlreadyRequest = false;
                else
                    newTransaction();
            }

        }

    }

    private void newTransaction() {

        boolean write = false;

        synchronized (MergingGraphRequestProcessor.this) {
            assert(!requestQueue.isEmpty());
            Object nextRequest = requestQueue.peekFirst().first;
            write = (nextRequest instanceof Write || nextRequest instanceof DelayedWrite);
        }

        if(write) {
            processor.asyncRequest(new RunnerWriteGraphRequest(), null);
        } else {
            processor.asyncRequest(new MergedRead());
        }

    }

    @Override
    public <T> void asyncRequest(AsyncMultiRead<T> request, AsyncMultiProcedure<T> procedure) {

//        System.out.println(this + " asyncRequest(ReadGraphRequest<QueryProcedure4<T>> request, QueryProcedure4<T> procedure)");

        if (requestSet.contains(request))
            return;

        Pair<Object, Object> pair = new Pair<Object, Object>(request, procedure);
        requestQueue.add(pair);
        requestSet.add(pair);

        if (!hasAlreadyRequest) {
            newTransaction();
            hasAlreadyRequest = true;
        } else {
            notify();
        }

    }

    @Override
    public synchronized <T> void asyncRequest(AsyncRead<T> request, AsyncProcedure<T> procedure) {

//        System.out.println(this + " asyncRequest(ReadGraphRequest<SingleQueryProcedure4<T>> request, SingleQueryProcedure4<T> procedure) " + this);

        if (requestSet.contains(request))
            return;

        Pair<Object, Object> pair = new Pair<Object, Object>(request, procedure);
        requestQueue.add(pair);
        requestSet.add(pair);

        if (!hasAlreadyRequest) {
            newTransaction();
            hasAlreadyRequest = true;
        } else {
//            System.out.println("notify " + this);
            notify();
        }

    }

    @Override
    public synchronized void asyncRequest(Write request, Consumer<DatabaseException> callback) {

//        System.out.println(this + " asyncRequest(WriteGraphRequest request)");

        if (requestSet.contains(request))
            return;

        Pair<Object, Object> pair = new Pair<Object, Object>(request, callback);
        requestQueue.add(pair);
        requestSet.add(pair);

        if (!hasAlreadyRequest) {
//            System.out.println("new transaction");
            newTransaction();
            hasAlreadyRequest = true;
        } else {
//            System.out.println("notify");
            notify();
        }

    }

    @Override
    public synchronized void asyncRequest(DelayedWrite request, Consumer<DatabaseException> callback) {

//        System.out.println(this + " asyncRequest(WriteGraphRequest request)");

        if (requestSet.contains(request))
            return;

        Pair<Object, Object> pair = new Pair<Object, Object>(request, callback);
        requestQueue.add(pair);
        requestSet.add(pair);

        if (!hasAlreadyRequest) {
//            System.out.println("new transaction");
            newTransaction();
            hasAlreadyRequest = true;
        } else {
//            System.out.println("notify");
            notify();
        }

    }

    @Override
    public synchronized void asyncRequest(WriteOnly request, Consumer<DatabaseException> callback) {

//        System.out.println(this + " asyncRequest(WriteGraphRequest request)");

        if (requestSet.contains(request))
            return;

        Pair<Object, Object> pair = new Pair<Object, Object>(request, callback);
        requestQueue.add(pair);
        requestSet.add(pair);

        if (!hasAlreadyRequest) {
//            System.out.println("new transaction");
            newTransaction();
            hasAlreadyRequest = true;
        } else {
//            System.out.println("notify");
            notify();
        }

    }

    @Override
    public <T> Collection<T> syncRequest(AsyncMultiRead<T> request, final AsyncMultiProcedure<T> procedure) {

        final DataContainer<Throwable> throwable = new DataContainer<Throwable>(null);

        // Queue the adapter up for execution.
        synchronized (request) {
            syncRequests.add(request);
            asyncRequest(request, procedure);
            if(syncRequests.contains(request)) {
                try {
                    //            	System.out.println("waiting " + request);
                    request.wait();
                } catch (InterruptedException e) {
                    throw new Error(e);
                }
            }
        }

        Throwable t = throwable.get();

        if(t != null) {
    		Logger.defaultLogError(t);
            throw new RuntimeException(t.getMessage());
        }
        
        return null;

    }

    @Override
    public <T> T syncRequest(AsyncRead<T> request, final AsyncProcedure<T> procedure) {

//        System.out.println("syncRequest(ReadGraphRequest<SingleQueryProcedure4<T>> request, SingleQueryProcedure4<T> procedure)");

        final DataContainer<T> result = new DataContainer<T>(null);
        final DataContainer<Throwable> throwable = new DataContainer<Throwable>(null);

        // Queue the adapter up for execution.
        synchronized (request) {

            syncRequests.add(request);
            asyncRequest(request, new AsyncProcedure<T>() {

                public void execute(AsyncReadGraph graph, T t) {
                    synchronized(result) {
                        result.set(t);
                    }
                    procedure.execute(graph, t);
                };

                @Override
                public void exception(AsyncReadGraph graph, Throwable t) {
                    throwable.set(t);
                }

                @Override
                public String toString() {
                    return procedure.toString();
                }

            });
            if(syncRequests.contains(request)) {
                try {
                    //            	System.out.println("waiting " + request);
                    request.wait();
                } catch (InterruptedException e) {
                    throw new Error(e);
                }
            }
        }

        Throwable t = throwable.get();

        if(t != null) {
    		Logger.defaultLogError(t);
            throw new RuntimeException(t.getMessage());
        }
        
        return result.get();

        //return result.get();

    }


    @Override
    public void syncRequest(Write request) {

//        System.out.println(MergingGraphRequestProcessor.this + " syncRequest(WriteGraphRequest)");

        SyncWriteRequestAdapter adapter = new SyncWriteRequestAdapter(request);

        asyncRequest(adapter, null);

        adapter.acquire();

        // Throw exception if one occurred.
        adapter.throwOrWrapException();

    }

    @Override
    public void syncRequest(WriteOnly request) {

//        System.out.println(MergingGraphRequestProcessor.this + " syncRequest(WriteGraphRequest)");

        SyncWriteRequestAdapter adapter = new SyncWriteRequestAdapter(request);

        // Queue the adapter up for execution.
        synchronized (adapter) {
            asyncRequest(adapter, null);
            try {
                adapter.wait();
            } catch (InterruptedException e) {
                throw new Error(e);
            }
        }

        // Throw exception if one occurred.
        adapter.throwOrWrapException();

    }

    @Override
    public Session getSession() {
        return processor.getSession();
    }

    @Override
    public String toString() {
        return "MergingGraphRequestProcessor[" + name + "]@" + System.identityHashCode(this) + " (based on " + processor + ")";
    }

    @Override
    public <T> void asyncRequest(AsyncRead<T> request) {

        asyncRequest(request, new ProcedureAdapter<T>() {

            @Override
            public void exception(Throwable t) {
	    		Logger.defaultLogError(t);
            }

        });

    }

    @Override
    public <T> void asyncRequest(AsyncRead<T> request, Procedure<T> procedure) {
    	asyncRequest(request, new NoneToAsyncProcedure<T>(procedure));
    }

    @Override
    public <T> void asyncRequest(AsyncMultiRead<T> request) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> void asyncRequest(AsyncMultiRead<T> request,
            MultiProcedure<T> procedure) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> T syncRequest(AsyncRead<T> request) {

        final DataContainer<Throwable> throwable = new DataContainer<Throwable>(null);
        final DataContainer<T> result = new DataContainer<T>();

        syncRequest(request, new AsyncProcedure<T>() {

            public void execute(AsyncReadGraph graph, T t) {
                result.set(t);
            }

            @Override
            public void exception(AsyncReadGraph graph, Throwable t) {
                throwable.set(t);
            }

        });

        Throwable t = throwable.get();

        if(t != null) {
    		Logger.defaultLogError(t);
            throw new RuntimeException(t.getMessage());
        }

        return result.get();

    }

    @Override
    public <T> T syncRequest(AsyncRead<T> request,
            Procedure<T> procedure) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> Collection<T> syncRequest(AsyncMultiRead<T> request) {

        final DataContainer<Throwable> throwable = new DataContainer<Throwable>(null);
        final ArrayList<T> result = new ArrayList<T>();

        syncRequest(request, new AsyncMultiProcedureAdapter<T>() {

            @Override
            public void execute(AsyncReadGraph graph, T t) {
                synchronized(result) {
                    result.add(t);
                }
            };

            @Override
            public void exception(AsyncReadGraph graph, Throwable t) {
                throwable.set(t);
            }

        });

        Throwable t = throwable.get();

        if(t != null) {
    		Logger.defaultLogError(t);
            throw new RuntimeException(t.getMessage());
        }

        return result;

    }

    @Override
    public <T> Collection<T> syncRequest(AsyncMultiRead<T> request,
            MultiProcedure<T> procedure) {
        throw new Error("Not implemented.");
    }

    @Override
    public <T> T syncRequest(Read<T> request) {

        final DataContainer<Throwable> throwable = new DataContainer<Throwable>(null);
        final DataContainer<T> result = new DataContainer<T>();


        syncRequest(request, new Procedure<T>() {

            public void execute(T t) {
                result.set(t);
            }

            @Override
            public void exception(Throwable t) {
                throwable.set(t);
            }

        });

        Throwable t = throwable.get();

        if(t != null) {
            throw new Error(t.getMessage());
        }


        return result.get();

    }

    @Override
    public <T> T syncRequest(Read<T> request,
            final AsyncProcedure<T> procedure) {

        final DataContainer<T> result = new DataContainer<T>(null);
        final DataContainer<Throwable> throwable = new DataContainer<Throwable>(null);

        // Queue the adapter up for execution.
        synchronized (request) {

            syncRequests.add(request);
            asyncRequest(request, new AsyncProcedure<T>() {

                public void execute(AsyncReadGraph graph, T t) {
                    synchronized(result) {
                        result.set(t);
                    }
                    procedure.execute(graph, t);
                };

                @Override
                public void exception(AsyncReadGraph graph, Throwable t) {
                    throwable.set(t);
                }

                @Override
                public String toString() {
                    return procedure.toString();
                }

            });
            if(syncRequests.contains(request)) {
                try {
                    //            	System.out.println("waiting " + request);
                    request.wait();
                } catch (InterruptedException e) {
                    throw new Error(e);
                }
            }
        }

        Throwable t = throwable.get();

        if(t != null) {
            throw new RuntimeException("Unexpected exception in MergingGraphRequestProcessor.syncRequest(Read, AsyncProcedure)", t);
        }
        
        return result.get();

    }

    @Override
    public <T> void asyncRequest(Read<T> request) {

        asyncRequest(request, new ProcedureAdapter<T>() {

            @Override
            public void exception(Throwable t) {
                Logger.defaultLogError(t);
            }

        });

    }

    @Override
    public synchronized <T> void asyncRequest(Read<T> request,
            AsyncProcedure<T> procedure) {

        if (requestSet.contains(request))
            return;

        Pair<Object, Object> pair = new Pair<Object, Object>(request, procedure);
        requestQueue.add(pair);
        requestSet.add(pair);

        if (!hasAlreadyRequest) {
            newTransaction();
            hasAlreadyRequest = true;
        } else {
            notify();
        }

    }

    @Override
    public <T> T syncRequest(Read<T> request, Procedure<T> procedure) {
        return syncRequest(request, new NoneToAsyncProcedure<T>(procedure));
    }

    @Override
    public <T> Collection<T> syncRequest(final MultiRead<T> request) throws DatabaseException {
        assert(request != null);

        final ArrayList<T> result = new ArrayList<T>();
        final DataContainer<Throwable> exception = new DataContainer<Throwable>();

        syncRequest(request, new AsyncMultiProcedureAdapter<T>() {

            @Override
            public void execute(AsyncReadGraph graph, T t) {
                synchronized(result) {
                    result.add(t);
                }
            }

            @Override
            public void exception(AsyncReadGraph graph, Throwable t) {
                exception.set(t);
            }

            @Override
            public String toString() {
                return "syncRequest(MultiRead) -> " + request;
            }

        });

        Throwable t = exception.get();
        if(t != null) {
            if(t instanceof DatabaseException) throw (DatabaseException)t;
            else throw new DatabaseException("Unexpected exception in ReadGraph.syncRequest(Read)", t);
        }

        return result;
    }

    @Override
    public <T> Collection<T> syncRequest(MultiRead<T> request, AsyncMultiProcedure<T> procedure)  {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public <T> Collection<T> syncRequest(MultiRead<T> request, MultiProcedure<T> procedure) {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public <T> void asyncRequest(Read<T> request, Procedure<T> procedure) {
        asyncRequest(request, new NoneToAsyncProcedure<T>(procedure));
    }

    @Override
    public <T> void asyncRequest(MultiRead<T> request) {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public <T> void asyncRequest(MultiRead<T> request, AsyncMultiProcedure<T> procedure) {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public <T> void asyncRequest(MultiRead<T> request, MultiProcedure<T> procedure) {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public void asyncRequest(Write r) {
        asyncRequest(r, null);
    }

    @Override
    public void asyncRequest(DelayedWrite r) {
        asyncRequest(r, null);
    }

    @Override
    public void asyncRequest(WriteOnly r) {
        asyncRequest(r, null);
    }

    @Override
    public <T> T getService(Class<T> api) {
        return getSession().getService(api);
    }

    @Override
    public <T> T peekService(Class<T> api) {
        return getSession().peekService(api);
    }

    @Override
    public boolean hasService(Class<?> api) {
        return getSession().hasService(api);
    }

    @Override
    public <T> void registerService(Class<T> api, T service) {
        getSession().registerService(api, service);
    }

//    @Override
//    public <T> T syncRequest(Read<T> arg0, AsyncListener<T> arg1) {
//        throw new UnsupportedOperationException("Not implemented.");
//    }

    @Override
    public <T> T syncRequest(Read<T> arg0, SyncListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> T syncRequest(Read<T> arg0, Listener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> T syncRequest(Read<T> arg0, SyncProcedure<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> T syncRequest(AsyncRead<T> arg0, AsyncListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> T syncRequest(AsyncRead<T> arg0, SyncListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> T syncRequest(AsyncRead<T> arg0, Listener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> T syncRequest(AsyncRead<T> arg0, SyncProcedure<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(Read<T> arg0, AsyncListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(Read<T> arg0, SyncListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(Read<T> arg0, Listener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(Read<T> arg0, SyncProcedure<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(AsyncRead<T> arg0, AsyncListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(AsyncRead<T> arg0, SyncListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(AsyncRead<T> arg0, Listener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(AsyncRead<T> arg0, SyncProcedure<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> Collection<T> syncRequest(MultiRead<T> arg0, AsyncMultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> Collection<T> syncRequest(MultiRead<T> arg0, SyncMultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> Collection<T> syncRequest(MultiRead<T> arg0, MultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> Collection<T> syncRequest(MultiRead<T> arg0, SyncMultiProcedure<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> Collection<T> syncRequest(AsyncMultiRead<T> arg0,
            AsyncMultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> Collection<T> syncRequest(AsyncMultiRead<T> arg0, SyncMultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> Collection<T> syncRequest(AsyncMultiRead<T> arg0,
            MultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> Collection<T> syncRequest(AsyncMultiRead<T> arg0, SyncMultiProcedure<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(MultiRead<T> arg0, AsyncMultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(MultiRead<T> arg0, SyncMultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(MultiRead<T> arg0, MultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(MultiRead<T> arg0, SyncMultiProcedure<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(AsyncMultiRead<T> arg0,
            AsyncMultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> void asyncRequest(AsyncMultiRead<T> arg0, SyncMultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(AsyncMultiRead<T> arg0,
            MultiListener<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");

    }

    @Override
    public <T> void asyncRequest(AsyncMultiRead<T> arg0, SyncMultiProcedure<T> arg1) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> void asyncRequest(ExternalRead<T> request, Procedure<T> procedure) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> T syncRequest(ExternalRead<T> request) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> T syncRequest(ExternalRead<T> request, Listener<T> procedure) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> T syncRequest(ExternalRead<T> request, Procedure<T> procedure) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> void asyncRequest(ExternalRead<T> request) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public <T> void asyncRequest(ExternalRead<T> request, Listener<T> procedure) {
        throw new UnsupportedOperationException("Not implemented.");
    }

    @Override
    public void syncRequest(DelayedWrite request) throws DatabaseException {
        throw new UnsupportedOperationException("Not implemented.");
    }

	@Override
	public <T> T syncRequest(WriteResult<T> request) throws DatabaseException {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> T syncRequest(DelayedWriteResult<T> request)
			throws DatabaseException {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> T syncRequest(WriteOnlyResult<T> r) throws DatabaseException {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> void asyncRequest(WriteResult<T> r, Procedure<T> procedure) {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> void asyncRequest(DelayedWriteResult<T> r, Procedure<T> procedure) {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> void asyncRequest(WriteOnlyResult<T> r, Procedure<T> procedure) {
		throw new UnsupportedOperationException();
	}
	
	@Override
	public Resource getRootLibrary() {
		return processor.getRootLibrary();
	}

	@Override
	public <T> void async(ReadInterface<T> r, Procedure<T> procedure) {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> void async(ReadInterface<T> r, AsyncProcedure<T> procedure) {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> void async(ReadInterface<T> r, SyncProcedure<T> procedure) {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> void async(ReadInterface<T> r, Listener<T> procedure) {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> void async(ReadInterface<T> r, AsyncListener<T> procedure) {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> void async(ReadInterface<T> r, SyncListener<T> procedure) {
		throw new UnsupportedOperationException();
	}
	
	@Override
	public <T> T sync(ReadInterface<T> r) throws DatabaseException {
		throw new UnsupportedOperationException();
	}
	
	@Override
	public <T> T sync(WriteInterface<T> r) throws DatabaseException {
		throw new UnsupportedOperationException();
	}
	
	@Override
	public <T> void async(WriteInterface<T> r, Procedure<T> procedure) {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> void async(WriteInterface<T> r) {
		throw new UnsupportedOperationException();
	}
	
	@Override
	public Object getModificationCounter() {
		throw new UnsupportedOperationException();
	}
	
}
