package org.simantics.db.common.issue;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

import org.simantics.databoard.Bindings;
import org.simantics.db.Issue;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.IndexRoot;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.layer0.Layer0;

public class AnonymousIssue implements Issue {

	public final Resource type;
	public final Resource[] contexts;

	public AnonymousIssue(Resource type, Resource ... contexts) {
		this.type = type;
		this.contexts = contexts;
	}

	public AnonymousIssue(Resource type, Collection<Resource> contexts) {
		this.type = type;
		this.contexts = contexts.toArray(new Resource[contexts.size()]);
	}

	public void writeAdditionalContext(WriteGraph graph, Resource issue) throws DatabaseException {

		IssueResource IR = IssueResource.getInstance(graph);

		// The main context
		graph.claim(issue, IR.Issue_HasContext, IR.Issue_HasContext_Inverse, contexts[0]);
		// A possible parent
		Layer0 L0 = Layer0.getInstance(graph);
		Resource parent = graph.getPossibleObject(contexts[0], L0.PartOf);
		if(parent != null) {
			graph.claim(issue, IR.Issue_HasContext, IR.Issue_HasContext_Inverse, parent);
		}

	}

	@Override
	public Resource write(WriteGraph graph, Resource source) throws DatabaseException {

		Layer0 L0 = Layer0.getInstance(graph);
		IssueResource IR = IssueResource.getInstance(graph);

		Resource model = graph.syncRequest(new IndexRoot(source), TransientCacheListener.<Resource>instance());

		Resource issue = graph.newResource();
		graph.claim(issue, L0.InstanceOf, null, type);
		graph.addLiteral(issue, L0.HasName, L0.NameOf, L0.String, UUID.randomUUID().toString(), Bindings.STRING);

		graph.claim(issue, IR.Issue_HasContexts, IR.Issue_HasContexts_Inverse, ListUtils.create(graph, L0.List, contexts));

		writeAdditionalContext(graph, issue);

		DateFormat format = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss");
		String created = format.format(Calendar.getInstance().getTime());
		graph.addLiteral(issue, IR.Issue_creationTime, IR.Issue_creationTime_Inverse, L0.String, created, Bindings.STRING);
		graph.claim(source, IR.IssueSource_Manages, IR.IssueSource_Manages_Inverse, issue);
		graph.claim(model, L0.ConsistsOf, L0.PartOf, issue);
		
		return issue;

	}

	@Override
	public String toString() {
		return Arrays.toString(contexts);
	}

	@Override
	public Resource getType() {
		return type;
	}

	@Override
	public Object getMainContext() {
		return contexts[0];
	}

	@Override
	public Set<Resource> getContexts() {
		HashSet<Resource> result = new HashSet<Resource>();
		for(Resource context : contexts) result.add(context);
		return result;
	}

}
