package org.simantics.databoard.serialization.impl;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.List;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer.NonRecursiveSerializer;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.util.Range;

public class IntArraySerializer extends NonRecursiveSerializer {

	Range length;
	Integer fixedLength, fixedSize;
	
	public IntArraySerializer(ArrayBinding binding)
	{
		ArrayType arrayType = (ArrayType) binding.type();
		this.length = arrayType.getLength();
		if (length!=null && length.getLower().equals(length.getUpper()) && length.getLower().getValue()!=null)
		{
			fixedLength = length.getLower().getValue().intValue();
			fixedSize = fixedLength * 4;
		}
	}
	
	@Override
	public Object deserialize(DataInput in)
	throws IOException 
	{
		int length = fixedSize != null ? fixedLength : in.readInt();		
		if (length<0) throw new SerializationException("Cannot use negative array length");
		assertRemainingBytes(in, length*4L);					
		
		int[] array = new int[length];
		for(int i=0;i<array.length;++i)
			array[i] = in.readInt();
		return array;
	}
	
	public Object deserializeToTry(DataInput in, List<Object> identities, Object obj) throws IOException
	{
		int length = fixedLength != null ? fixedLength : in.readInt();
		int[] array = (int[]) obj;
		if (length!=array.length) array = new int[ length ];
		assertRemainingBytes(in, length*4L);					
		
		for (int i=0; i<array.length;i++)
			array[i] = in.readInt();
		
		return array;
	}

	@Override
	public void deserializeTo(DataInput in, Object obj) throws IOException {
		int length = fixedLength != null ? fixedLength : in.readInt();
		float[] array = (float[]) obj;
		if (length!=array.length) throw new SerializationException("primitive array is size immutable");
		assertRemainingBytes(in, length*4L);					
		for (int i=0; i<array.length;i++)
			array[i] = in.readFloat();
	}		
	
	@Override
	public void skip(DataInput in)
	throws IOException {
		int length = fixedSize != null ? fixedLength : in.readInt();			
		in.skipBytes(length*4);
	}
	
	@Override
	public void serialize(DataOutput out, Object obj)
	throws IOException 
	{
		int[] array = (int[])obj;
		if (fixedSize==null) 
			out.writeInt(array.length);
		for(int f : array)
			out.writeInt(f);
	}

	@Override
	public Integer getConstantSize() {
		return fixedSize;
	}

	@Override
	public int getSize(Object obj) {
		if (fixedSize!=null) return fixedSize;
		int[] array = (int[])obj;			
		return 4 + 4 * array.length;
	}

	@Override
	public int getMinSize() {
		return fixedSize != null ? fixedSize : 4;
	}
	
}