package org.simantics.databoard.serialization.impl;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.List;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer.NonRecursiveSerializer;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.util.Range;

public class ByteArraySerializer extends NonRecursiveSerializer {

	Range length;
	Integer fixedLength, fixedSize;
	
	public ByteArraySerializer(ArrayBinding binding)
	{
		ArrayType arrayType = (ArrayType) binding.type();
		this.length = arrayType.getLength();
		if (length!=null && length.getLower().equals(length.getUpper()) && length.getLower().getValue()!=null)
		{
			fixedLength = length.getLower().getValue().intValue();
			fixedSize = fixedLength * 1;
		}
	}
	
	@Override
	public Object deserialize(DataInput in)
	throws IOException 
	{
		int length = fixedSize != null ? fixedLength : in.readInt();
		if (length<0) throw new SerializationException("Cannot use negative array length");
		assertRemainingBytes(in, length);			
		
		byte[] array = new byte[length];
		in.readFully(array);
		return array;
	}
	
	@Override
	public void deserializeTo(DataInput in, Object obj) throws IOException {
		try {
			int length = fixedLength != null ? fixedLength : in.readInt();			
			byte[] array = (byte[]) obj;
			if (length!=array.length) throw new BindingException("primitive array is size immutable");
			assertRemainingBytes(in, length);			
			in.readFully(array);
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}
	
	public Object deserializeToTry(DataInput in, List<Object> identities, Object obj) throws IOException
	{
		int length = fixedLength != null ? fixedLength : in.readInt();
		byte[] array = (byte[]) obj;
		if (length!=array.length) obj = new byte[ length ];
		assertRemainingBytes(in, length);			
		in.readFully(array);		
		return array;
	}

	@Override
	public void skip(DataInput in)
	throws IOException {
		int length = fixedSize != null ? fixedLength : in.readInt();			
		in.skipBytes( length );
	}
	
	@Override
	public void serialize(DataOutput out, Object obj)
	throws IOException 
	{
		byte[] array = (byte[])obj;
		if (fixedSize==null) 
			out.writeInt(array.length);
		out.write(array);
	}

	@Override
	public Integer getConstantSize() {
		return fixedSize;
	}

	@Override
	public int getSize(Object obj) {
		if (fixedSize!=null) return fixedSize;
		byte[] array = (byte[])obj;			
		return 4 + array.length;
	}
	
	@Override
	public int getMinSize() {
		return fixedSize != null ? fixedSize : 4;
	}
	
}