/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.MapBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.reflection.VoidBinding;
import org.simantics.databoard.type.MapType;
import org.simantics.databoard.type.RecordType;

/**
 * Binds java.util.Set to Map(T, {})
 * 
 * This binding accepts all java.util.Set instances, but instantiates
 * java.util.TreeSet objects.
 *
 * @author Reino Ruusu <reino.ruusu@vtt.fi>
 */
@SuppressWarnings("unchecked")
public class DefaultSetBinding extends MapBinding {
	
	public DefaultSetBinding(MapType mapType, Binding elementBinding) {
		super(mapType, elementBinding, VoidBinding.VOID_BINDING);
	}
	
	public DefaultSetBinding(Binding elementBinding) {
		super(new MapType(elementBinding.type(), RecordType.VOID_TYPE), elementBinding, VoidBinding.VOID_BINDING);
	}

	@Override
	public void clear(Object set) throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		_set.clear();
	}

	@Override
	public boolean containsKey(Object set, Object key) throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		return _set.contains(key);
	}

	@Override
	public boolean containsValue(Object set, Object value)
			throws BindingException {
		return false;
	}

    @Override
    public Object create() throws BindingException {
        return new TreeSet<Object>( keyBinding );
    }

	public Object create(Set<?> initialSet) throws BindingException {
	    return initialSet;
	}

	@Override
	public Object create(Map<?, ?> initialMap) throws BindingException {
		Set<Object> result = new TreeSet<Object>( keyBinding );
		result.addAll(initialMap.keySet());
		return result;
	}

	@Override
	public Object create(Object[] keys, Object[] values)
	throws BindingException {		
		Set<Object> result = new TreeSet<Object>( keyBinding );
		for (int i=0; i<keys.length; i++) {
			result.add( keys[i] );
		}
		return result;
	}
	
	@Override
	public Object create(List<Object> keys, List<Object> values) {
		Set<Object> result = new TreeSet<Object>( keyBinding );
		for (int i=0; i<keys.size(); i++)
			result.add(keys.get(i));
		return result;
	}	

	@Override
	public Object get(Object set, Object key) throws BindingException {
		return null;
	}
	
    @Override
    public <K, V> void getAll(Object setFrom, Map<K, V> to) throws BindingException {
		Map<K, V> _to = (Map<K, V>) to;
		Set<K> _setFrom = (Set<K>) setFrom;
		for (K k : _setFrom)
			_to.put(k, null);
	}

	@Override
	public void getAll(Object setFrom, Object[] keys, Object[] values)
			throws BindingException {
		Set<Object> _setFrom = (Set<Object>) setFrom;
		int i=0;
		for (Object k : _setFrom) {
			keys[i] = k;
			values[i] = null;
			i++;
		}
	}

	@Override
	public Object[] getKeys(Object set) throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		return _set.toArray(new Object[_set.size()]);
	}

	@Override
	public void getKeys(Object set, Set<Object> keys) throws BindingException {
		Set<Object> s = (Set<Object>) set;
		keys.addAll(s);
	}
	
	@Override
	public Object[] getValues(Object set) throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		return new Object[_set.size()];
	}
	
	@Override
	public int count(Object src, Object from, boolean fromInclusive, Object end, boolean endInclusive) throws BindingException {
	    if (src instanceof TreeSet)
            return new TreeSetBinding(keyBinding).count(src, from, fromInclusive, end, endInclusive);
	    else
	        return new HashSetBinding(keyBinding).count(src, from, fromInclusive, end, endInclusive);
	}
	
	@Override
	public int getEntries(Object src, Object from, boolean fromInclusive, Object end, boolean endInclusive, ArrayBinding dstKeyArrayBinding, Object dstKeyArray, ArrayBinding dstValueArrayBinding, Object dstValueArray, int limit) throws BindingException {
		return 0;
	}

	@Override
	public void put(Object set, Object key, Object value)
			throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		if (value!=null) throw new BindingException("Cannot put non-null to a Set");
		_set.add(key);
	}

	public void putAll(Object setTo, Set<?> from) {
		Set<Object> _set = (Set<Object>) setTo;
		_set.addAll(from);
	}
	
	@Override
	public <K,V> void putAll(Object setTo, Map<K,V> from) throws BindingException {
		Set<Object> _set = (Set<Object>) setTo;
		_set.addAll(from.keySet());
	}

	@Override
	public Object remove(Object set, Object key) throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		_set.remove(key);
		return null;
	}

	@Override
	public int size(Object set) throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		return _set.size();
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof Set;
	}

	@Override
	public Object getCeilingKey(Object set, Object key) {
        if (set instanceof TreeSet)
            return new TreeSetBinding(keyBinding).getCeilingKey(set, key);
        else
            return new HashSetBinding(keyBinding).getCeilingKey(set, key);
	}

	@Override
	public Object getFirstKey(Object set) {
        if (set instanceof TreeSet)
            return new TreeSetBinding(keyBinding).getFirstKey(set);
        else
            return new HashSetBinding(keyBinding).getFirstKey(set);
	}

	@Override
	public Object getFloorKey(Object set, Object key) {
        if (set instanceof TreeSet)
            return new TreeSetBinding(keyBinding).getFloorKey(set, key);
        else
            return new HashSetBinding(keyBinding).getFloorKey(set, key);
	}

	@Override
	public Object getHigherKey(Object set, Object key) {
        if (set instanceof TreeSet)
            return new TreeSetBinding(keyBinding).getHigherKey(set, key);
        else
            return new HashSetBinding(keyBinding).getHigherKey(set, key);
	}

	@Override
	public Object getLastKey(Object set) {
        if (set instanceof TreeSet)
            return new TreeSetBinding(keyBinding).getLastKey(set);
        else
            return new HashSetBinding(keyBinding).getLastKey(set);
	}

	@Override
	public Object getLowerKey(Object set, Object key) {
        if (set instanceof TreeSet)
            return new TreeSetBinding(keyBinding).getLowerKey(set, key);
        else
            return new HashSetBinding(keyBinding).getLowerKey(set, key);
	}

}

