/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.charts.query;

import java.util.UUID;

import org.simantics.charts.ontology.ChartResource;
import org.simantics.charts.ui.ChartItemLabelRule;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Databoard;
import org.simantics.databoard.binding.Binding;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.PossibleTypedParent;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleActiveExperiment;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.subscription.SubscriptionItemLabel;
import org.simantics.trend.configuration.Scale;
import org.simantics.trend.configuration.TrendItem;
import org.simantics.trend.configuration.TrendItem.DrawMode;

/**
 * This query returns TrendItem description
 * 
 * @author toni.kalajainen
 */
public class TrendItemQuery extends ResourceRead<TrendItem> {

	public TrendItemQuery(Resource resource) {
		super(resource);
	}

	@SuppressWarnings("unused")
	@Override
	public TrendItem perform(ReadGraph graph) throws DatabaseException {
        ChartResource CHART = ChartResource.getInstance(graph);
        G2DResource G2D = G2DResource.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);
        Layer0 L0 = Layer0.getInstance(graph);

    	TrendItem item = new TrendItem();

    	item.label = ChartItemLabelRule.INSTANCE.getLabel_(graph, resource);        	

        scale: {
            Double low = null;
            Double high = null;
            Resource sc = graph.getPossibleObject(resource, CHART.Chart_Item_ScaleMode);
            if (CHART.ScaleMode_AutoScale.equals(sc)) {
                item.scale = new Scale.Auto();
            } else if (CHART.ScaleMode_ManualScale.equals(sc)) {
                Scale.Manual manualScale = new Scale.Manual();
                item.scale = manualScale;
                low = graph.getPossibleRelatedValue(resource, CHART.Chart_Item_ScaleMode_Min, Bindings.DOUBLE);
                high = graph.getPossibleRelatedValue(resource, CHART.Chart_Item_ScaleMode_Max, Bindings.DOUBLE);
                manualScale.min = low == null ? 0 : low;
                manualScale.max = high == null ? 120 : high;
            } else {
                item.scale = new Scale.Auto();
            }
        }
        
        index: {
           	Integer index = graph.getPossibleRelatedValue(resource, CHART.Chart_Item_Index);
            item.index = index != null ? index : 1;
        }

        renderer: {
            item.renderer = TrendItem.Renderer.Analog;
            Resource rend = graph.getPossibleObject(resource, CHART.Chart_Item_Renderer);
            if (CHART.Renderer_Binary.equals(rend)) {
                item.renderer = TrendItem.Renderer.Binary;
            }
        }

        drawmode: {
            Resource dm = graph.getPossibleObject(resource, CHART.Chart_Item_DrawMode);
            if (CHART.DrawMode_DeviationAndAverage.equals(dm)) {
                item.drawMode = DrawMode.DeviationAndAverage;
            } else if (CHART.DrawMode_Average.equals(dm)) {
            	item.drawMode = DrawMode.Average;
//            } else if (CHART.DrawMode_Median.equals(dm)) {
//            	item.drawMode = DrawMode.Median;
            } else if (CHART.DrawMode_Sample.equals(dm)) {
            	item.drawMode = DrawMode.Sample;
//            } else if (CHART.DrawMode_DeviationAndMedian.equals(dm)) {
//            	item.drawMode = DrawMode.DeviationAndMedian;
            } else if (CHART.DrawMode_DeviationAndSample.equals(dm)) {
            	item.drawMode = DrawMode.DeviationAndSample;
            } else if (CHART.DrawMode_DeviationAndLine.equals(dm)) {
            	item.drawMode = DrawMode.DeviationAndLine;
            } else if (CHART.DrawMode_Line.equals(dm)) {
            	item.drawMode = DrawMode.Line;
            } else if (CHART.DrawMode_Deviation.equals(dm)) {
            	item.drawMode = DrawMode.Deviation;
            }
        }

        visualAttributes: {
            Boolean hidden = graph.getPossibleRelatedValue(resource, CHART.Chart_Item_hidden, Bindings.BOOLEAN);
            item.hidden = hidden != null ? hidden : false;
            item.customStrokeWidth = DiagramGraphUtil.getPossibleRelatedValue(graph, resource, G2D.HasStrokeWidth, Float.class, null);
            float[] color = graph.getPossibleRelatedValue(resource, G2D.HasColor, Bindings.FLOAT_ARRAY);
            if (color != null && color.length >= 3)
                item.customColor = color;
        }

        item.variableId = "";
        item.groupItemId = "";
        Resource subscriptionItem = graph.getPossibleObject(resource, CHART.Chart_Item_HasSubscriptionItem);
        Resource subscription = null;
        if (subscriptionItem!=null) {
        	item.simpleLabel = graph.getPossibleRelatedValue(subscriptionItem, L0.HasLabel, Bindings.STRING);
            subscription = graph.syncRequest( new PossibleTypedParent(subscriptionItem, MOD.Subscription) );
        }
        if (item.simpleLabel==null) item.simpleLabel = "";
        if (subscriptionItem != null) 
        	SubscriptionItem: { 
		
		        label2: {
        			item.groupItemId = graph.getPossibleRelatedValue(subscriptionItem, L0.HasName, Bindings.STRING);
        			if (item.groupItemId == null) break SubscriptionItem;

        			Binding rviBinding = graph.getService(Databoard.class).getBindingUnchecked( RVI.class );
		            RVI rvi = graph.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_VariableId, rviBinding);
		            if (rvi == null) break SubscriptionItem;        
		            //variablePersistentId = rvi.toString();
		            
					Variable configuration = Variables.getPossibleConfigurationContext(graph, subscription);
					if (configuration == null) break SubscriptionItem;
					
					Resource activeExperiment = graph.syncRequest(new PossibleActiveExperiment(subscription));
					if (activeExperiment != null) {
						Variable experimentVariable = null;
						try {
							experimentVariable = Variables.switchRealization(graph, configuration, activeExperiment);
						} catch (DatabaseException e) {
							experimentVariable = Variables.switchPossibleContext(graph, configuration, activeExperiment);
						}
						if (experimentVariable != null) {
							configuration = experimentVariable;
						}
					}

					item.variableId = item.variableReference = rvi.toPossibleString(graph, configuration);
		            if (item.variableId == null) break SubscriptionItem;

		            if (item.simpleLabel.isEmpty()) {
		            	item.simpleLabel = SubscriptionItemLabel.removeVariablePrefixPath( item.variableId );
		            }
		            /*
		            SimulationResource SIM = SimulationResource.getInstance(graph);
		            Resource model = graph.syncRequest( new PossibleTypedParent(resource, SIM.Model) );		            
		            Variable base = null;
		            if (model!=null) base = Variables.getConfigurationContext(graph, model);
		        	if (base != null) {
		        		item.label2 = rvi.asString(graph, base);
		        		if (!item.label2.isEmpty() && item.label2.charAt(0)=='/') item.label2 = item.label2.substring(1);
		        	} else {
		        		item.label2 = rvi.toString();
		        	}*/
		        }
		        
		    	subscription: {
		            if (subscription!=null) {
		                item.groupId = graph.getPossibleRelatedValue(subscription, L0.HasName, Bindings.STRING);
		            } else {
		                item.groupId = UUID.randomUUID().toString();
		            }
		            if (item.label == null || item.label.isEmpty()) {
		            	item.label = item.simpleLabel;
		        		if (!item.label.isEmpty() && item.label.charAt(0)=='/') item.label = item.label.substring(1);
		            }
		    	}
		        
		        
		        unit: {
		            item.unit = graph.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_Unit, Bindings.STRING);
		//            if (item.unit != null && !item.unit.isEmpty())
		//                item.label += " [" + item.unit + "]";
		        }
        } 
        
        if (item.groupItemId==null) item.groupItemId = "";
        if (item.variableId==null) item.variableId = "";
        if (item.groupId==null) item.groupId = "";
        if (item.unit==null) item.unit = "";
        
        return item;
	}

}
