/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.BuiltinKeys.ImagerKey;
import org.simantics.browsing.ui.DataSource;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.common.EvaluatorData.Transformer;
import org.simantics.browsing.ui.common.imagers.ContainerImager;
import org.simantics.browsing.ui.content.Imager;
import org.simantics.browsing.ui.content.ImagerFactory;
import org.simantics.browsing.ui.graph.impl.MissingImageDescriptor;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.ui.icons.ImageDescriptorProvider;
import org.simantics.utils.datastructures.cache.ProvisionException;
import org.simantics.utils.ui.ErrorLogger;

/**
 * @author Antti Villberg
 */
public class AdapterImagerFactory implements ImagerFactory {

    final private Transformer transformer;
    final private HashMap<String, Image> constants;
    final private String resourceColumn;

    public AdapterImagerFactory() {
        this(null);
    }

    public AdapterImagerFactory(Transformer transformer) {
        this(transformer, null, new HashMap<String, Image>(4));
    }

    public AdapterImagerFactory(Transformer transformer, String resourceColumn, HashMap<String, Image> constants) {
        this.transformer = transformer;
        this.constants = constants;
        this.resourceColumn = resourceColumn;
    }

    private Resource getInputResource(final NodeContext context) {
        Object input = context.getConstant(BuiltinKeys.INPUT);
        if (transformer != null) {
            input = transformer.transform(input);
        }

        if (input instanceof Resource)
            return (Resource) context.getConstant(BuiltinKeys.INPUT);
        else if (input instanceof IAdaptable)
            return (Resource) (((IAdaptable) input).getAdapter(Resource.class));
        return null;
    }


    @SuppressWarnings("unchecked")
    @Override
    public Imager create(final PrimitiveQueryUpdater updater, final NodeContext context, final ImagerKey key) {
        @SuppressWarnings("rawtypes")
        final ContainerImager result = new ContainerImager((HashMap<String, Image>) constants.clone());
        result.setImage(resourceColumn, MissingImageDescriptor.getInstance());

        final DataSource<ReadGraph> source = updater.getDataSource(ReadGraph.class);
        final Resource inputResource = getInputResource(context);

        source.schedule(graph -> {
            ImageDescriptor image = null;
            Collection<Resource> types = Collections.emptyList();
            try {
                types = graph.getPrincipalTypes(inputResource);
//                System.out.println("AdapterImagerFactory " + source.adapt(inputResource, String.class));
//                for(Resource r : types) {
//                    System.out.println("-" + source.adapt(r, String.class));
//                }
            } catch (DatabaseException e1) {
                ErrorLogger.defaultLogError(e1);
            }
            for (Resource type : types) {
                try {
                    ImageDescriptorProvider provider = graph.getPossibleAdapter(type, ImageDescriptorProvider.class);
                    if (provider != null) {
                        try {
                            ImageDescriptor descriptor = provider.get();
                            if (descriptor != null)
                                image = descriptor;
                            //System.out.println("*" + image);
                        } catch (ProvisionException pe) {
                            ErrorLogger.defaultLogError(pe);
                        }
                        break;
                    }
                } catch (DatabaseException e) {
                    ErrorLogger.defaultLogError(e);
                }
            }

            result.setImage(resourceColumn, image);
            updater.scheduleReplace(context, key, result);
        });

        return result;
    }

}
