package org.simantics.browsing.ui.nattable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.jface.viewers.IPostSelectionProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.nebula.widgets.nattable.NatTable;
import org.eclipse.nebula.widgets.nattable.coordinate.PositionCoordinate;
import org.eclipse.nebula.widgets.nattable.layer.ILayerListener;
import org.eclipse.nebula.widgets.nattable.layer.cell.ILayerCell;
import org.eclipse.nebula.widgets.nattable.layer.event.ILayerEvent;
import org.eclipse.nebula.widgets.nattable.selection.SelectionLayer;
import org.eclipse.nebula.widgets.nattable.selection.event.CellSelectionEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Event;
import org.simantics.utils.datastructures.MapList;
import org.simantics.utils.ui.AdaptionUtils;

public class NatTableSelectionAdaptor implements ISelectionProvider, IPostSelectionProvider, ILayerListener {
	NatTable natTable;
	SelectionLayer selectionLayer;
	GETreeData treeData;
	StructuredSelection selection;
	
	private List<ISelectionChangedListener> selectionListeners = new ArrayList<>();
	private List<ISelectionChangedListener> postSelectionListeners = new ArrayList<>();
	private List<SelectionListener> selListeners = new ArrayList<>();
	
	public NatTableSelectionAdaptor(NatTable natTable, SelectionLayer selectionLayer, GETreeData treeData) {
		this.natTable = natTable;
		this.selectionLayer = selectionLayer;
		this.natTable.addLayerListener(this);
		this.treeData = treeData;
	}
	
	@Override
	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		selectionListeners.add(listener);
	}
	
	@Override
	public void removeSelectionChangedListener(ISelectionChangedListener listener) {
		selectionListeners.remove(listener);
	}
	
	@Override
	public void addPostSelectionChangedListener(ISelectionChangedListener listener) {
		postSelectionListeners.add(listener);
	}
	
	@Override
	public void removePostSelectionChangedListener(ISelectionChangedListener listener) {
		postSelectionListeners.remove(listener);
	}
	
	public void addSelectionListener(SelectionListener listener) {
		selListeners.add(listener);
	}
	
	public void removeSelectionListener(SelectionListener listener) {
		selListeners.remove(listener);
	}
	
	@Override
	public ISelection getSelection() {
		return selection;
	}
	
	@Override
	public void setSelection(ISelection selection) {
		if (!(selection instanceof StructuredSelection))
			throw new IllegalArgumentException("Selection must be structured selection");
		if (selection.isEmpty()) {
			selectionLayer.clear(false);
			natTable.redraw();
			return;
		}
		List<RowSelectionItem> rowItems = new ArrayList<>(AdaptionUtils.adaptToCollection(selection, RowSelectionItem.class));
		if (rowItems.size() > 0) {
		
			setSelectionExternal(rowItems);
			return;
		}
		Collection<TreeNode> nodes = AdaptionUtils.adaptToCollection(selection, TreeNode.class);
		if (nodes.size() > 0) {
			List<RowSelectionItem> selected = new ArrayList<>();
			int allCols[] = new int[selectionLayer.getColumnCount()];
			for (int i = 0; i < allCols.length; i++)
				allCols[i] = i;
			for (TreeNode n : nodes) {
				selected.add(new RowSelectionItem(n, n.listIndex, allCols));
			}
			setSelectionExternal(selected);
			return;
		}
		
	}
	
	private void setSelectionExternal(List<RowSelectionItem> items) {
		selectionLayer.clear(true);
		for (RowSelectionItem item : items) {
			for (int c : item.columnIndex) {
				int r = selectionLayer.getRowPositionByIndex(item.rowIndex);
				selectionLayer.selectCell(c, r, false, true);
			}
		}
		selection = new StructuredSelection(items);
		fireEvents();
	}
	
	
	private List<Point> selectedCells = new ArrayList<Point>();
	
	@Override
	public void handleLayerEvent(ILayerEvent event) {
		if (event instanceof CellSelectionEvent) {
			evaluateSeletedCells();
		} 
	}
	
	/**
	 * Processes current selection to data indices.
	 */
	private void evaluateSeletedCells() {
		selectedCells.clear();
		for (PositionCoordinate pc : selectionLayer.getSelectedCellPositions()) {
			ILayerCell cell = pc.getLayer().getCellByPosition(pc.columnPosition, pc.rowPosition);
			selectedCells.add(new Point(cell.getColumnIndex(), cell.getRowIndex()));
		}
		MapList<Integer, Integer> rowMap = new MapList<>();
		for (Point p : selectedCells) {
			rowMap.add(p.y, p.x);
		}
		List<RowSelectionItem> selectionItems = new ArrayList<>(rowMap.getKeySize());
		for (Integer row : rowMap.getKeys()) {
			List<Integer> cols = rowMap.getValues(row);
			int col[] = new int[cols.size()];
			for (int i = 0; i < col.length; i++)
				col[i] = cols.get(i);
			selectionItems.add(new RowSelectionItem(treeData.getDataAtIndex(row), row, col));
		}
		this.selection = new StructuredSelection(selectionItems);
		fireEvents();
	}
	
	private void fireEvents() {
		for (ISelectionChangedListener l : selectionListeners) {
			l.selectionChanged(new SelectionChangedEvent(this, selection));
		}
		for (ISelectionChangedListener l : postSelectionListeners) {
			l.selectionChanged(new SelectionChangedEvent(this, selection));
		}
		Event evt = new Event();
		evt.widget = natTable;
		evt.display = natTable.getDisplay();
		evt.data = selection;
		for (SelectionListener l : selListeners) {
			SelectionEvent sel = new SelectionEvent(evt);
			l.widgetSelected(sel);
		}
	}
	
	public List<Point> getSelectedCells() {
		return selectedCells;
	}

}
