package org.simantics.annotation.ui.editor;

import org.eclipse.ui.IEditorInput;
import org.simantics.NameLabelUtil;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ui.features.EditorNamingService2;
import org.simantics.ui.workbench.IEditorNamingService2;
import org.simantics.ui.workbench.IResourceEditorInput;

/**
 * Tries to:
 * <ul>
 * <li>resolve annotation property relation name from the input</li>
 * <li>add (model-name) suffix to editor tooltip to tell apart editors with same
 * title. The goal tooltip format is: path/input-name (model-name)</li>
 * </ul>
 * 
 * @author Tuukka Lehtonen
 */
public class AnnotationTypeEditorNamingService extends EditorNamingService2 implements IEditorNamingService2 {

    @Override
    public String getName(ReadGraph graph, String editorId, IEditorInput in) throws DatabaseException {
        if (!(in instanceof IResourceEditorInput))
            return "";

        IResourceEditorInput input = (IResourceEditorInput) in;
        Resource r = input.getResource();
        Layer0 L0 = Layer0.getInstance(graph);
        Resource property = graph.getPossibleObject(r, L0.HasRange_Inverse);
        return limitedName(graph, NameLabelUtil.modalName(graph, property != null ? property : r), input);
    }

    @Override
    public String getToolTipText(ReadGraph graph, String editorId, IEditorInput input) throws DatabaseException {
        return getTooltip(graph, editorId, (IResourceEditorInput) input, new StringBuilder()).toString();
    }

    private StringBuilder getTooltip(ReadGraph graph, String editorId, IResourceEditorInput input, StringBuilder sb) throws DatabaseException {
        if (input.getResource() != null) {
            Resource r = input.getResource();

            Resource model = graph.syncRequest(new PossibleModel(r));
            if (model != null) {
                String uri = graph.getPossibleURI(input.getResource());
                if (uri != null) {
                    String modelURI = graph.getPossibleURI(model);
                    String path = uri.replace(modelURI + "/", "");
                    // Remove last uri segment which is the type name (UUID)
                    path = path.replaceFirst("/[^/]*$", "/");
                    sb.append(URIStringUtils.unescape(path));
                }
                sb.append(getName(graph, editorId, input));

                // Append model label in parenthesis
                String modelLabel = NameLabelUtil.modalName(graph, model);
                sb.append(" (").append(modelLabel).append(")");
            }
        }

        return sb;
    }

}