/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.collect.set;

import com.koloboke.collect.ContainerFactory;
import com.koloboke.collect.Equivalence;
import com.koloboke.compile.KolobokeSet;
import java.util.function.Consumer;

import javax.annotation.Nonnull;

import java.util.Iterator;


/**
 * An immutable factory of {@code ShortSet}s.
 *
 * 
 * @param <F> the concrete factory type which extends this interface
 * @see ShortSet
 * @see KolobokeSet @KolobokeSet
 */
public interface ShortSetFactory<
        F extends ShortSetFactory<F>> extends ContainerFactory<F> {


    

    

    

    


    /**
     * Constructs a new empty mutable set of the {@linkplain #getDefaultExpectedSize() default
     * expected size}.
     *
     
     * @return a new empty mutable set
     */
    @Nonnull
     ShortSet newMutableSet();

    /**
     * Constructs a new empty mutable set of the given expected size.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty mutable set of the given expected size
     */
    @Nonnull
     ShortSet newMutableSet(int expectedSize);

    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Iterable<Short> elements, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
     ShortSet newMutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5, int expectedSize);

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Iterator<Short> elements, int expectedSize);

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull short[] elements, int expectedSize);

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Short[] elements, int expectedSize);

    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Iterable<Short> elements);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
     ShortSet newMutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5);

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Iterator<Short> elements);

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            );

    

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull short[] elements);

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     ShortSet newMutableSet(@Nonnull Short[] elements);


    /**
     * Constructs a new mutable singleton set of the given element.
     *
     * @param e1 the sole element
    
     * @return a new mutable singleton set of the given element
     */
    @Nonnull
     ShortSet newMutableSetOf(short e1);

    /**
     * Constructs a new mutable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new mutable set of the two specified elements
     */
    @Nonnull
     ShortSet newMutableSetOf(short e1, short e2);

    /**
     * Constructs a new mutable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new mutable set of the three specified elements
     */
    @Nonnull
     ShortSet newMutableSetOf(short e1, short e2, short e3);

    /**
     * Constructs a new mutable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new mutable set of the four specified elements
     */
    @Nonnull
     ShortSet newMutableSetOf(short e1, short e2, short e3,
            short e4);

    /**
     * Constructs a new mutable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new mutable set of the specified elements
     */
    @Nonnull
     ShortSet newMutableSetOf(short e1, short e2, short e3,
            short e4, short e5, short... restElements);

    /**
     * Constructs a new empty updatable set of the {@linkplain #getDefaultExpectedSize() default
     * expected size}.
     *
     
     * @return a new empty updatable set
     */
    @Nonnull
     ShortSet newUpdatableSet();

    /**
     * Constructs a new empty updatable set of the given expected size.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty updatable set of the given expected size
     */
    @Nonnull
     ShortSet newUpdatableSet(int expectedSize);

    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Iterable<Short> elements, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
     ShortSet newUpdatableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5, int expectedSize);

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Iterator<Short> elements, int expectedSize);

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull short[] elements, int expectedSize);

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Short[] elements, int expectedSize);

    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Iterable<Short> elements);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
     ShortSet newUpdatableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5);

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Iterator<Short> elements);

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            );

    

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull short[] elements);

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     ShortSet newUpdatableSet(@Nonnull Short[] elements);


    /**
     * Constructs a new updatable singleton set of the given element.
     *
     * @param e1 the sole element
    
     * @return a new updatable singleton set of the given element
     */
    @Nonnull
     ShortSet newUpdatableSetOf(short e1);

    /**
     * Constructs a new updatable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new updatable set of the two specified elements
     */
    @Nonnull
     ShortSet newUpdatableSetOf(short e1, short e2);

    /**
     * Constructs a new updatable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new updatable set of the three specified elements
     */
    @Nonnull
     ShortSet newUpdatableSetOf(short e1, short e2, short e3);

    /**
     * Constructs a new updatable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new updatable set of the four specified elements
     */
    @Nonnull
     ShortSet newUpdatableSetOf(short e1, short e2, short e3,
            short e4);

    /**
     * Constructs a new updatable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new updatable set of the specified elements
     */
    @Nonnull
     ShortSet newUpdatableSetOf(short e1, short e2, short e3,
            short e4, short e5, short... restElements);

    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Iterable<Short> elements, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
     ShortSet newImmutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5, int expectedSize);

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Iterator<Short> elements, int expectedSize);

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull short[] elements, int expectedSize);

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Short[] elements, int expectedSize);

    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Iterable<Short> elements);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
     ShortSet newImmutableSet(@Nonnull Iterable<Short> elems1,
            @Nonnull Iterable<Short> elems2,
            @Nonnull Iterable<Short> elems3,
            @Nonnull Iterable<Short> elems4,
            @Nonnull Iterable<Short> elems5);

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Iterator<Short> elements);

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull
            Consumer<com.koloboke.function.ShortConsumer> elementsSupplier
            );

    

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull short[] elements);

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
     ShortSet newImmutableSet(@Nonnull Short[] elements);


    /**
     * Constructs a new immutable singleton set of the given element.
     *
     * @param e1 the sole element
    
     * @return a new immutable singleton set of the given element
     */
    @Nonnull
     ShortSet newImmutableSetOf(short e1);

    /**
     * Constructs a new immutable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new immutable set of the two specified elements
     */
    @Nonnull
     ShortSet newImmutableSetOf(short e1, short e2);

    /**
     * Constructs a new immutable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new immutable set of the three specified elements
     */
    @Nonnull
     ShortSet newImmutableSetOf(short e1, short e2, short e3);

    /**
     * Constructs a new immutable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new immutable set of the four specified elements
     */
    @Nonnull
     ShortSet newImmutableSetOf(short e1, short e2, short e3,
            short e4);

    /**
     * Constructs a new immutable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new immutable set of the specified elements
     */
    @Nonnull
     ShortSet newImmutableSetOf(short e1, short e2, short e3,
            short e4, short e5, short... restElements);
}
