/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import junit.framework.TestCase;

import org.junit.Test;
import org.simantics.databoard.util.Limit;
import org.simantics.databoard.util.Range;
import org.simantics.databoard.util.RangeException;

public class TestRange extends TestCase {

	@Test
	public void testParse() {
		
		// Test Limits
		assertEquals ( Limit.exclusive(5.0), Limit.exclusive(5L ) );
		assertEquals ( Limit.inclusive(5.0), Limit.inclusive(5L ) );
		assertEquals ( Limit.exclusive(5L ), Limit.exclusive(5L ) );
		assertEquals ( Limit.inclusive(5L ), Limit.inclusive(5L ) );
		assertEquals ( Limit.exclusive(5.0), Limit.exclusive(5.0) );
		assertEquals ( Limit.inclusive(5.0), Limit.inclusive(5.0) );
		assertNotSame( Limit.inclusive(5.0), Limit.exclusive(5L ) );
		assertNotSame( Limit.exclusive(5.0), Limit.inclusive(5L ) );
		assertNotSame( Limit.inclusive(5.0), Limit.inclusive(6.0) );
		assertNotSame( Limit.inclusive(5L ), Limit.inclusive(6L ) );
		
		
		// assert lower < upper
		try {
			Range.create(100, 50, true, true);
			fail();
		} catch (IllegalArgumentException e) {}					
		
		// assert lower < upper
		try {
			Range.create(100., 50., true, true);
			fail();
		} catch (IllegalArgumentException e) {}		
		
		// assert lower < upper
		try {
			Range.create(100.f, 50.f, true, true);
			fail();
		} catch (IllegalArgumentException e) {}			
		
		// toString
		assertEquals( "[0..100]", Range.create(0, 100, true, true).toString() );
		assertEquals( "[0..100)", Range.create(0, 100, true, false).toString() );
		assertEquals( "(0..100]", Range.create(0, 100, false, true).toString() );
		assertEquals( "[..]", new Range(Limit.nolimit(), Limit.nolimit()).toString() );
		assertEquals( "[0..)", Range.create(0, null, true, false).toString() );
		assertEquals( "(0..)", Range.create(0, null, false, false).toString() );
		assertEquals( "(..100)", Range.create(null, 100, false, false).toString() );
		assertEquals( "0", Range.create(0, 0, true, true).toString() );
		assertEquals( "()", Range.create(100, 100, false, false).toString() );
		assertEquals( "100", Range.create(100, 100, true, true).toString() );
		assertEquals( "-100", Range.create(-100, -100, true, true).toString() );

		// Parse
		try {
			assertEquals( Range.valueOf("[0..100]"), Range.create(0, 100, true, true) );
			assertEquals( Range.valueOf("[0..100)"), Range.create(0, 100, true, false) );
			assertEquals( Range.valueOf("(0..100]"), Range.create(0, 100, false, true) );
			assertEquals( Range.valueOf("[..]"), new Range(Limit.nolimit(), Limit.nolimit()) );
			assertEquals( Range.valueOf("[0..)"), Range.create(0, null, true, false) );
			assertEquals( Range.valueOf("(0..)"), Range.create(0, null, false, false) );
			assertEquals( Range.valueOf("(..100)"), Range.create(null, 100, false, false) );
			assertEquals( Range.valueOf("0"), Range.create(0, 0, true, true) );
			assertEquals( Range.valueOf("()"), Range.create(100, 100, false, false) );
			assertEquals( Range.valueOf("100"), Range.create(100, 100, true, true) );
			
			assertEquals( Range.valueOf("[0.0E0..100.0E0]"), Range.create(0, 100, true, true) );
			assertEquals( Range.valueOf("[0.0E0..100.0E0)"), Range.create(0, 100, true, false) );
			assertEquals( Range.valueOf("(0.0E0..100.0E0]"), Range.create(0, 100, false, true) );
			assertEquals( Range.valueOf("[..]"), new Range(Limit.nolimit(), Limit.nolimit()) );
			assertEquals( Range.valueOf("[0.0E0..)"), Range.create(0, null, true, false) );
			assertEquals( Range.valueOf("(0.0E0..)"), Range.create(0, null, false, false) );
			assertEquals( Range.valueOf("(..100.0E0)"), Range.create(null, 100, false, false) );
			assertEquals( Range.valueOf("()"), Range.create(100, 100, false, false) );
			assertEquals( Range.valueOf("100.0E0"), Range.create(100, 100, true, true) );
			
			assertEquals( Range.valueOf("[-100..0]"), Range.create(-100, 0, true, true) );
			assertEquals( Range.valueOf("[-100..0)"), Range.create(-100, 0, true, false) );
			assertEquals( Range.valueOf("(-100..0]"), Range.create(-100, 0, false, true) );
			assertEquals( Range.valueOf("[-100..0)"), Range.create(-100, 0, true, false) );
			assertEquals( Range.valueOf("(-100..)"), Range.create(-100, null, false, false) );
			assertEquals( Range.valueOf("(..-100)"), Range.create(null, -100, false, false) );
			assertEquals( Range.valueOf("()"), Range.create(-100, -100, false, false) );
			assertEquals( Range.valueOf("-100.0"), Range.create(-100, -100, true, true) );
			
			assertEquals( Range.valueOf("[-100.0E0..-0.0E0]"), Range.create(-100, 0, true, true) );
			assertEquals( Range.valueOf("[..-100)"), Range.create(null, -100, false, false) );
			
		} catch (RangeException e) {
			fail(e.getMessage());
		}
		
		
		// Parse-fail
		try {
			Range.valueOf("(0]");
			fail();
		} catch (RangeException e) {} 

		try {
			Range.valueOf("(0)");
			fail();
		} catch (RangeException e) {}
		
		try {
			Range.valueOf("[0");
			fail();
		} catch (RangeException e) {} 

		try {
			Range.valueOf("[0]");
			fail();
		} catch (RangeException e) {} 
		
		try {
			Range.valueOf("[]");
			fail();
		} catch (RangeException e) {} 
		
		try {
			Range.valueOf("");
			fail();
		} catch (RangeException e) {}
		
		try {
			Range.valueOf("{0..4]");
			fail();
		} catch (RangeException e) {} 
		
		try {
			Range.valueOf("[5..4]");
			fail();
		} catch (IllegalArgumentException e) {} catch (RangeException e) { fail(); }
		
		try {
			Range.valueOf("[0,4]");
			fail();
		} catch (RangeException e) {} 
		
		try {
			Range.valueOf("[0 .. 4]");
			fail();
		} catch (RangeException e) {} 
		
		try {
			Range.valueOf("[x..4]");
			fail();
		} catch (RangeException e) {} 
		
		// inRange()
		try {
			assertFalse( Range.valueOf("[0..100]").contains(-10) );
			assertTrue ( Range.valueOf("[0..100]").contains(0) );
			assertTrue ( Range.valueOf("[0..100]").contains(50) );
			assertTrue ( Range.valueOf("[0..100]").contains(100) );
			assertFalse( Range.valueOf("[0..100]").contains(150) );

			assertFalse( Range.valueOf("(0..100)").contains(-10) );
			assertFalse( Range.valueOf("(0..100)").contains(0) );
			assertTrue ( Range.valueOf("(0..100)").contains(50) );
			assertFalse( Range.valueOf("(0..100)").contains(100) );
			assertFalse( Range.valueOf("(0..100)").contains(150) );

			assertFalse( Range.valueOf("(0..)").contains(-10) );
			assertFalse( Range.valueOf("(0..)").contains(0) );
			assertTrue ( Range.valueOf("(0..)").contains(50) );
			assertTrue ( Range.valueOf("(0..)").contains(100) );
			assertTrue ( Range.valueOf("(0..)").contains(150) );

			assertFalse( Range.valueOf("[0..)").contains(-10) );
			assertTrue ( Range.valueOf("[0..)").contains(0) );
			assertTrue ( Range.valueOf("[0..)").contains(50) );
			assertTrue ( Range.valueOf("[0..)").contains(100) );
			assertTrue ( Range.valueOf("[0..)").contains(150) );
			
			assertTrue ( Range.valueOf("(..100)").contains(-10) );
			assertTrue ( Range.valueOf("(..100)").contains(0) );
			assertTrue ( Range.valueOf("(..100)").contains(50) );
			assertFalse( Range.valueOf("(..100)").contains(100) );
			assertFalse( Range.valueOf("(..100)").contains(150) );			

			assertTrue ( Range.valueOf("(..100]").contains(-10) );
			assertTrue ( Range.valueOf("(..100]").contains(0) );
			assertTrue ( Range.valueOf("(..100]").contains(50) );
			assertTrue ( Range.valueOf("(..100]").contains(100) );
			assertFalse( Range.valueOf("(..100]").contains(150) );			
			
			assertTrue ( Range.valueOf("[..]").contains(-10) );
			assertTrue ( Range.valueOf("[..]").contains(0) );
			assertTrue ( Range.valueOf("[..]").contains(50) );
			assertTrue ( Range.valueOf("[..]").contains(100) );
			assertTrue ( Range.valueOf("[..]").contains(150) );

			assertFalse( Range.valueOf("()").contains(-10) );
			assertFalse( Range.valueOf("()").contains(0) );
			assertFalse( Range.valueOf("()").contains(50) );
			
			assertFalse( Range.valueOf("100").contains(50) );
			assertTrue ( Range.valueOf("100").contains(100) );
			assertFalse( Range.valueOf("100").contains(150) );
			
			assertFalse( Range.valueOf("100").contains(50) );
			assertTrue ( Range.valueOf("100").contains(100) );
			assertFalse( Range.valueOf("100").contains(150) );

			
		} catch (RangeException e) {
			fail(e.getMessage());
		}
		
	}

}

