/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.datastructures.hints;

/**
 * HintTracker is a helper class for tracking any hints of a single
 * IHintContext.
 * 
 * <p>
 * The main purpose of this class is to enable very easy switching of the
 * tracked hint context. Only a call to the {@link #track(IHintContext)} method
 * is required. All hint listeners registered into the tracked will
 * automatically be switched to listen to the new context instead.
 * </p>
 * 
 * <p>
 * The tracker is implemented using HintStack but the
 * {@link #addHintContext(IHintContext, int)} and
 * {@link #removeHintContext(IHintContext)} methods are disabled by throwing
 * {@link UnsupportedOperationException} in both. This is to prevent invalid use
 * of the class as it is only meant for tracking a single context and nothing
 * more. Otherwise this class functions exactly like {@link IHintStack} and
 * {@link IHintObservable} are specified to work.
 * </p>
 * 
 * <p>
 * The implementation is thread-safe.
 * </p>
 * 
 * <p>
 * Always be sure to {@link #untrack()} any IHintContext that you're tracking
 * after the tracker is no longer needed. Otherwise you will most likely have
 * invalid listeners within the tracked IHintContext resulting in erroneous
 * behavior.
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public class HintTracker extends HintStack implements IHintTracker {
    
    private IHintContext ctx;

    public HintTracker() {
    }

    public HintTracker(IHintContext ctx) {
        addHintContext(ctx, 0);
    }

    /* (non-Javadoc)
     * @see org.simantics.utils.datastructures.hints.IHintTracker#track(org.simantics.utils.datastructures.hints.IHintContext)
     */
    public synchronized final void track(IHintContext ctx) {
        IHintContext prevCtx = this.ctx;
        if (ctx == prevCtx)
            return;
        this.ctx = ctx;

        if (ctx != null) {
            if (ctx.equals(prevCtx)) {
                // remove+add order is necessary in this case.
                // add+remove would fail since the hint context would
                // already be found in the hint stack.
                super.removeHintContext(prevCtx);
                super.addHintContext(ctx, 0);
            } else {
                super.addHintContext(ctx, 0);
                if (prevCtx != null)
                    super.removeHintContext(prevCtx);
            }
        } else {
            if (prevCtx != null)
                super.removeHintContext(prevCtx);
        }
    }
    
    /* (non-Javadoc)
     * @see org.simantics.utils.datastructures.hints.IHintTracker#untrack()
     */
    public final void untrack() {
        track(null);
    }
    
    @Override
    public final void addHintContext(IHintContext hints, int priority) {
        throw new UnsupportedOperationException();
    }

    @Override
    public final boolean removeHintContext(IHintContext hints) {
        throw new UnsupportedOperationException();
    }

}