package org.simantics.ui.workspace.tracker.internal.contributions;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.ui.model.application.MApplication;
import org.eclipse.e4.ui.model.application.ui.MUIElement;
import org.eclipse.e4.ui.model.application.ui.menu.MToolControl;
import org.eclipse.e4.ui.workbench.modeling.EModelService;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.PlatformUI;
import org.simantics.filesystem.services.sizetracker.DirectorySizeService;
import org.simantics.filesystem.services.sizetracker.SizeTracker;
import org.simantics.ui.workspace.tracker.internal.Activator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 * @since 1.31.0
 */
public class WorkspaceSizeTrackerContribution {

	private Logger logger = LoggerFactory.getLogger(WorkspaceSizeTrackerContribution.class);

	private SizeTracker workspaceSizeTracker;

	@PostConstruct
	void createControls(Composite parent, MToolControl toolControl) {
		String trackSize = System.getProperty("org.simantics.filesystem.services.sizeTracker", null);
		if (trackSize != null && trackSize.equalsIgnoreCase("false"))
			return;
		Path ws = getWorkspacePath();
		if (ws != null && !Files.isDirectory(ws))
			return;
		DirectorySizeService dss = Activator.getDefault().getDirectorySizeService();
		if (dss == null)
			return;

		try {
			workspaceSizeTracker = dss.track(ws);
			new WorkspaceSizeTrackerTrim(parent, toolControl, workspaceSizeTracker, Activator.getDefault().getPreferenceStore());
		} catch (IOException e) {
			logger.error("Failed to start workspace size tracking for {}", ws, e);
		}
	}

	@PreDestroy
	void dispose() {
		try (SizeTracker t = workspaceSizeTracker) {
		} catch (IOException e) {
			logger.error("Failed to shutdown workspace size tracking for {}", workspaceSizeTracker, e);
		} finally {
			workspaceSizeTracker = null;
		}
	}

	private static Path getWorkspacePath() {
		IPath ip = Platform.getLocation();
		return ip != null ? ip.toFile().toPath() : null;
	}

	public static void showTracker(boolean show) {
		IEclipseContext context = PlatformUI.getWorkbench().getService(IEclipseContext.class);
		if (context == null)
			return;

		EModelService modelService = context.get(EModelService.class);
		MApplication app = context.get(MApplication.class);
		if (modelService == null || app == null)
			return;

		MUIElement element = modelService.find("org.simantics.ui.workspace.tracker.toolcontrol.0", app);
		if (element != null && element.isToBeRendered() != show) {
			element.setToBeRendered(show);
			Object widget = element.getWidget();
			if (widget instanceof Control) {
				((Control) widget).getShell().layout(null, SWT.ALL | SWT.CHANGED | SWT.DEFER);
			}
		}
	}

}