/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.event.view.contribution;

import gnu.trove.map.hash.TObjectDoubleHashMap;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.simantics.browsing.ui.model.children.ChildRule;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.QueryControl;
import org.simantics.event.ontology.EventResource;
import org.simantics.event.view.Constants;
import org.simantics.event.view.preference.EventPrefs;
import org.simantics.layer0.Layer0;
import org.simantics.operation.Layer0X;
import org.simantics.simulation.ontology.SimulationResource;

/**
 * A child rule that looks for all event logs within the active models of a
 * project. The rule takes into account the preference setting for showing hidden
 * event logs.
 * 
 * @author Tuukka Lehtonen
 */
public enum ProjectEventsRule implements ChildRule {

    INSTANCE;

    public static ProjectEventsRule get() {
        return INSTANCE;
    }

    private static class Prefs {
        public final boolean hideInfoEvents;
        public final boolean hideWarningEvents;
        public final boolean hideReturnEvents;
        public final boolean showHiddenEvents;
        public final boolean showOnlyMilestones;
        public final boolean showOnlyActiveEvents;
        public Prefs(
                boolean hideInfoEvents,
                boolean hideWarningEvents,
                boolean hideReturnEvents,
                boolean showHiddenEvents,
                boolean showOnlyMilestones,
                boolean showOnlyActiveEvents) {
            this.hideInfoEvents = hideInfoEvents;
            this.hideWarningEvents = hideWarningEvents;
            this.hideReturnEvents = hideReturnEvents;
            this.showHiddenEvents = showHiddenEvents;
            this.showOnlyMilestones = showOnlyMilestones;
            this.showOnlyActiveEvents = showOnlyActiveEvents;
        }

        public static Prefs read(ReadGraph graph, Resource project) throws DatabaseException {
            return new Prefs(
                    EventPrefs.hideInfoEvents(graph, project),
                    EventPrefs.hideWarningEvents(graph, project),
                    EventPrefs.hideReturnEvents(graph, project),
                    EventPrefs.showHiddenEvents(graph, project),
                    EventPrefs.showOnlyMilestones(graph, project),
                    EventPrefs.showOnlyActiveEvents(graph, project));
        }
    }

    public Collection<?> getChildrenImpl(ReadGraph graph, Resource project, Prefs prefs, List<Resource> eventLogs) throws DatabaseException {

        Layer0 L0 = Layer0.getInstance(graph);

        EventResource EVENT = EventResource.getInstance(graph);

        final boolean hideInfoEvents = prefs.hideInfoEvents;
        final boolean hideWarningEvents = prefs.hideWarningEvents;
        final boolean hideReturnEvents = prefs.hideReturnEvents;
        final boolean showHiddenEvents = prefs.showHiddenEvents;
        final boolean showOnlyMilestones = prefs.showOnlyMilestones;
        final boolean showOnlyActiveEvents= prefs.showOnlyActiveEvents;

        ArrayList<Resource> allEvents = new ArrayList<Resource>();
        for (Resource log : eventLogs) {
        	for (Resource slice : graph.getObjects(log, L0.ConsistsOf)) {
        		Collection<Resource> events = graph.getObjects(slice, L0.ConsistsOf);
        		allEvents.addAll(events);
        	}
        }

        List<Object> result = new ArrayList<Object>();
        final TObjectDoubleHashMap<Object> times = new TObjectDoubleHashMap<Object>();
        final Double defaultTime = 0.0;

        for (Resource event : allEvents) {
            if (showOnlyMilestones) {
                boolean isMilestone = graph.hasStatement(event, EVENT.Milestone);
                if (!isMilestone)
                    continue;
            }

            if (!showHiddenEvents && graph.hasStatement(event, EVENT.Hidden))
                continue;

            boolean isReturnEvent = hideReturnEvents || showOnlyActiveEvents
                    ? graph.hasStatement(event, EVENT.ReturnEvent) : false;

            // Skip all return events if thus preferred.
            if (hideReturnEvents && isReturnEvent) {
                continue;
            }

            // Skip all return events and starting events that have been returned,
            // if thus preferred. Also skip events that are defined non-returnable.
            if (showOnlyActiveEvents
                    && (isReturnEvent
                            || graph.hasStatement(event, EVENT.Returns)
                            || graph.hasStatement(event, EVENT.ReturnedBy)
                            || graph.hasStatement(event, EVENT.NoReturn)))
            {
                continue;
            }

            // Filter by event type severity
            Resource eventType = graph.getPossibleObject(event, EVENT.Event_type);
            if (eventType != null) {
                Integer severity = graph.getPossibleRelatedValue(eventType, EVENT.EventType_severity);
                if (severity != null) {
                    if (hideInfoEvents && severity >= Constants.EVENT_SEVERITY_INFO
                            && severity < (Constants.EVENT_SEVERITY_INFO + Constants.EVENT_SEVERITY_STEP))
                        continue;
                    if (hideWarningEvents && severity >= Constants.EVENT_SEVERITY_WARNING
                            && severity < (Constants.EVENT_SEVERITY_WARNING + Constants.EVENT_SEVERITY_STEP))
                        continue;
                }
            }

            double timeNumeric = EventLabelRule.getTimestamp(graph, event, defaultTime);
            times.put(event, timeNumeric);
            result.add(event);
        }

        Collections.sort(result, new Comparator<Object>() {
            @Override
            public int compare(Object o1, Object o2) {
                double t1 = times.get(o1);
                double t2 = times.get(o2);
                return Double.compare(t1, t2);
            }
        });

        return result;

    }
    
    @Override
    public Collection<?> getChildren(ReadGraph graph, Object parent) throws DatabaseException {

        Layer0X L0X = Layer0X.getInstance(graph);
        SimulationResource SIMU = SimulationResource.getInstance(graph);
        EventResource EVENT = EventResource.getInstance(graph);

        Resource project = (Resource) parent;

        List<Resource> eventLogs = new ArrayList<Resource>();
        for (Resource activeModel : graph.syncRequest(new ObjectsWithType(project, L0X.Activates, SIMU.Model))) {
            for (Resource eventLog : graph.syncRequest(new ObjectsWithType(activeModel, EVENT.HasEventLog, EVENT.EventLog))) {
                if (!graph.hasStatement(eventLog, EVENT.Hidden)) {
                    eventLogs.add(eventLog);
                    graph.getRelatedValue(eventLog, EVENT.HasModificationCounter, Bindings.INTEGER);
                }
            }
        }

        QueryControl qc = graph.getService(QueryControl.class);

        return getChildrenImpl(qc.getIndependentGraph(graph), project, Prefs.read(graph, project), eventLogs);

    }

    @Override
    public Collection<?> getParents(ReadGraph graph, Object child) throws DatabaseException {
        if (!(child instanceof Resource))
            return Collections.emptyList();
        Layer0 L0 = Layer0.getInstance(graph);
        return graph.getObjects((Resource) child, L0.PartOf);
    }

    @Override
    public boolean isCompatible(Class<?> contentType) {
        return contentType.equals(Resource.class);
    }

}
