package org.simantics.db.layer0.validation;

import gnu.trove.set.hash.THashSet;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.simantics.db.Issue;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.Functions;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.SubgraphExtent.ExtentStatus;
import org.simantics.db.layer0.util.DomainProcessor3;
import org.simantics.db.layer0.util.ModelTransferableGraphSourceRequest;
import org.simantics.layer0.Layer0;

public class ValidationUtils {

    /**
     * Performs L0.Constraint validations for the specified resource and puts
     * the resulting issues in the specified {@link Issue} set. If the specified
     * issue set is <code>null</code>, a new set will be created and returned.
     * Otherwise the method function will return the same set as its return
     * value.
     * 
     * @param graph
     *            database read access handle
     * @param r
     *            resource to validate against ontology-defined constraints
     * @param result
     *            a set for the resulting issues or <code>null</code> to
     *            allocate a new set if necessary
     * @return the discovered set of issues in <code>r</code>
     * @throws DatabaseException
     */
    public static Set<Issue> validateConstraints(ReadGraph graph, Resource r, Set<Issue> result) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);

        for (Resource constraint : graph.syncRequest(new GetConstraints(r))) {
            try {
                Resource function = graph.getSingleObject(constraint, L0.Constraint_Validator);
                @SuppressWarnings("unchecked")
                List<Issue> issues = (List<Issue>)Functions.exec(graph, function, graph, r);
                if (issues != null && !issues.isEmpty()) {
                    if (result == null)
                        result = new THashSet<Issue>();
                    result.addAll(issues);
                }
            } catch (Throwable t) {
                Logger.defaultLogError(t);
            }
        }

        return result != null ? result : Collections.<Issue>emptySet();
    }

    public static Set<Issue> validateConstraints(ReadGraph graph, Resource r) throws DatabaseException {
        return validateConstraints(graph, r, null);
    }

    public static Set<Issue> validateConstraintsForDomain(ReadGraph graph, Resource r) throws DatabaseException {
        Set<Issue> result = null;

        DomainProcessor3 dp = ModelTransferableGraphSourceRequest.getDomainOnly(graph, null, r);
        for(Map.Entry<Resource,ExtentStatus> status : dp.getStatus().entrySet()) {
            if(ExtentStatus.INTERNAL.equals(status.getValue())) {
                Resource rr = status.getKey();
                Set<Issue> issues = validateConstraints(graph, rr, result);
                if (!issues.isEmpty())
                    result = issues;
            }
        }

        return result != null ? result : Collections.<Issue>emptySet();
    }

}
