/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import gnu.trove.map.hash.TObjectIntHashMap;

import java.util.concurrent.Semaphore;

import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.common.exception.DebugException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.procedure.ListenerBase;

public class URIToResource extends StringQuery<InternalProcedure<Integer>> {

//    public ArrayList<InternalProcedure<Integer>> procs = null;
    
    private URIToResource(final String id) {
        super(id);
    }
    
    final static URIToResource entry(final QueryProcessor provider, final String id) {
        return (URIToResource)provider.uriToResourceMap.get(id);
    }

    final static void runner(ReadGraphImpl graph, final String id, CacheEntry parent, final ListenerBase listener, final InternalProcedure<Integer> procedure) {

    	QueryProcessor processor = graph.processor;
    	
        URIToResource entry = (URIToResource)processor.uriToResourceMap.get(id); 
        if(entry == null) {

        	entry = new URIToResource(id);
        	entry.setPending();
        	entry.clearResult(processor.querySupport);
        	entry.putEntry(processor);

        	processor.performForEach(graph, entry, parent, listener, procedure);
            
        } else {
        	
            if(entry.isPending()) {
                synchronized(entry) {
                    if(entry.isPending()) {
                        throw new IllegalStateException();
//                    	if(entry.procs == null) entry.procs = new ArrayList<InternalProcedure<Integer>>();
//                        entry.procs.add(procedure);
//                        processor.registerDependencies(graph, entry, parent, listener, procedure, false);
//                        return;
                    }
                }
            }
            
            processor.performForEach(graph, entry, parent, listener, procedure);
            
        }
        
    }
    
    final public static void queryEach(ReadGraphImpl graph, final String id, final CacheEntry parent, final ListenerBase listener, final InternalProcedure<Integer> procedure) {
        
        assert(id != null);
        
        if(graph.parent == null && listener == null) {
        	URIToResource.computeForEach(graph, id, null, procedure);
        } else {
            runner(graph, id, parent, listener, procedure);
        }
         
    }
     
    @Override
    public URIToResource getEntry(QueryProcessor provider) {
        return provider.uriToResourceMap.get(id);
    }
    
    @Override
    public void putEntry(QueryProcessor provider) {
        provider.uriToResourceMap.put(id, this);
    }

    @Override
    final public void removeEntry(QueryProcessor provider) {
        provider.uriToResourceMap.remove(id);
    }

    private void lookup(ReadGraphImpl graph, final QueryProcessor processor, final InternalProcedure<Integer> procedure, final String namespace, final String name) {
        
        NamespaceIndex.queryEach(graph, namespace, processor, this, null, new InternalProcedure<TObjectIntHashMap<String>>() {

            @Override
            public void execute(ReadGraphImpl graph, TObjectIntHashMap<String> index) {

                if(index != null) {
                    int result = index.get(name);
                    if(result != 0) {
                        addOrSet(graph, processor, result);
                        procedure.execute(graph, result);
                        return;
                    }
                }
                
                addOrSet(graph, processor, new Integer(0));
                procedure.execute(graph, new Integer(0));

            }

            @Override
            public void exception(ReadGraphImpl graph, Throwable t) {
                except(t);
                procedure.exception(graph, t);
            }

        });

    }

    private static void lookup(ReadGraphImpl graph, final URIToResource entry, final InternalProcedure<Integer> procedure, final String namespace, final String name) {
        
        NamespaceIndex.queryEach(graph, namespace, graph.processor, entry, null, new InternalProcedure<TObjectIntHashMap<String>>() {

            @Override
            public void execute(ReadGraphImpl graph, TObjectIntHashMap<String> index) {

                if(index != null) {
                    int result = index.get(name);
                    if(result != 0) {
                        if(entry != null) entry.addOrSet(graph, graph.processor, result);
                        procedure.execute(graph, result);
                        return;
                    }
                }
                
                Integer zero = 0;
                if(entry != null) entry.addOrSet(graph, graph.processor, zero);
                procedure.execute(graph, zero);

            }

            @Override
            public void exception(ReadGraphImpl graph, Throwable t) {
                if(entry != null) entry.except(t);
                procedure.exception(graph, t);
            }

        });

    }
    
    @Override
    public void computeForEach(ReadGraphImpl graph, final QueryProcessor processor, final InternalProcedure<Integer> procedure) {
        
//    	new Exception("URIToResource " + id).printStackTrace();
    	
        if("http://".equals(id) || "http:/".equals(id)) {
            
            addOrSet(graph, processor, processor.getRootLibrary());
            procedure.execute(graph, processor.getRootLibrary());

        } else {
            
            final String[] parts = URIStringUtils.splitURI(id);
            if (parts != null) {
                lookup(graph, processor, procedure, parts[0], parts[1]);
            } else {
                lookup(graph, processor, procedure, "http://", id.replaceFirst("http://", ""));
            }

        }
        
    }

    final private static void computeForEach(ReadGraphImpl graph, String id, final URIToResource entry, final InternalProcedure<Integer> procedure) {
        
        if("http://".equals(id) || "http:/".equals(id)) {
            
        	QueryProcessor processor = graph.processor;
            if(entry != null) entry.addOrSet(graph, processor, processor.getRootLibrary());
            procedure.execute(graph, processor.getRootLibrary());

        } else {
            
            final String[] parts = URIStringUtils.splitURI(id);
            if (parts != null) {
                lookup(graph, entry, procedure, parts[0], parts[1]);
            } else {
                lookup(graph, entry, procedure, "http://", id.replaceFirst("http://", ""));
            }

        }
        
    }
    
    public void addOrSet(ReadGraphImpl graph, QueryProcessor provider, Integer result) {

        assert(isPending());

//        ArrayList<InternalProcedure<Integer>> p = null;

        synchronized(this) {

            setResult(result);
            setReady();
//            p = procs;
//            procs = null;
            
        }

//        if(p != null)
//        	for(InternalProcedure<Integer> proc : p) proc.execute(graph, result);
        
    }
    
    @Override
    public String toString() {
        return "URIToResource[" + id + "]";
    }

    @Override
    public void performFromCache(ReadGraphImpl graph, QueryProcessor provider, InternalProcedure<Integer> procedure) {
        
        assert(isReady());
        
    	if(handleException(graph, procedure)) return;
        
        if(isExcepted()) {
        	procedure.exception(graph, (Throwable)statusOrException);
        } else {
            procedure.execute(graph, (Integer)getResult());
        }
        
    }
    
    @Override
    public void recompute(ReadGraphImpl graph, QueryProcessor provider) {
        
        final Semaphore s = new Semaphore(0);
        
        computeForEach(graph, provider, new InternalProcedure<Integer>() {

            @Override
            public void execute(ReadGraphImpl graph, Integer result) {
                s.release();
            }
            
            @Override
            public void exception(ReadGraphImpl graph, Throwable t) {
                if(DebugException.DEBUG) new DebugException(t).printStackTrace();
                throw new Error("Error in recompute.", t);
            }

        });
        
    	while(!s.tryAcquire()) {
    		provider.resume(graph);
    	}
        
    }
	
}
