/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicBoolean;

import org.simantics.db.RelationInfo;
import org.simantics.db.common.exception.DebugException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.impl.procedure.TripleIntProcedureAdapter;
import org.simantics.db.procedure.ListenerBase;
import org.simantics.db.request.RequestFlags;

final public class Statements extends CollectionBinaryQuery<TripleIntProcedure> {
	
//	public ArrayList<TripleIntProcedure> procs = null;
	
    public Statements(final int r1, final int r2) {
        super(r1, r2);
    }

    final static Statements entry(final QueryProcessor processor, final int r1, final int r2) {
        
        return (Statements)processor.statementsMap.get(id(r1,r2));

    }
    
	final static Collection<Statements> entries(final QueryProcessor processor, final int r1) {
		return processor.statementsMap.values(r1);
	}
    
    final static void runner(ReadGraphImpl graph, final int r1, final int r2, CacheEntry parent, final ListenerBase listener, final TripleIntProcedure procedure) {
        
    	QueryProcessor processor = graph.processor;

        Statements entry = (Statements)processor.statementsMap.get(id(r1,r2));
        if(entry == null) {
        	
        	entry = new Statements(r1, r2);
        	entry.setPending();
        	entry.clearResult(processor.querySupport);
        	entry.putEntry(processor);
        	
        	processor.performForEach(graph, entry, parent, listener, procedure);
            
        } else {
        	
            if(entry.isPending()) {
            	synchronized(entry) {
                    if(entry.isPending()) {
                        throw new IllegalStateException();
//                        if(entry.procs == null) entry.procs = new ArrayList<TripleIntProcedure>();
//                    	entry.procs.add(procedure);
//                    	processor.registerDependencies(graph, entry, parent, listener, procedure, false);
//                    	return;
                    }
            	}
            }
            
            processor.performForEach(graph, entry, parent, listener, procedure);
            
        }

    }
    
    final public static void queryEach(ReadGraphImpl graph, final int r1, final int r2, final QueryProcessor provider, final CacheEntry parent, final ListenerBase listener, final TripleIntProcedure procedure) {
        
    	assert(r1 != 0);
    	assert(r2 != 0);
    	
        if(parent == null && listener == null) {
        	Statements.computeForEach(graph, r1, r2, null, procedure);
        	return;
        }
        
        runner(graph, r1, r2, parent, listener, procedure);
         
    }

    @Override
    public BinaryQuery<TripleIntProcedure> getEntry(QueryProcessor provider) {
        return provider.statementsMap.get(id);
    }
 	
 	@Override
 	public void putEntry(QueryProcessor provider) {
 	    provider.statementsMap.put(id, this);
 	}

 	@Override
 	final public void removeEntry(QueryProcessor provider) {
        provider.statementsMap.remove(id);
 	}
 	
    final static TripleIntProcedure NOPT = new TripleIntProcedure() {


		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

		@Override
		public void execute(ReadGraphImpl graph, int s, int p, int o) {
		}

		@Override
		public void finished(ReadGraphImpl graph) {
		}
    	
    };
 	
    final static private IntArray getAssertionMap(ReadGraphImpl graph, final int r1, final int r2, final Statements entry) {
    	
        class AssertionMapProc implements IntProcedure {
        	
        	boolean first = true;

        	private IntArray result;

        	public void addStatement(int s, int p, int o) {
        		
        		if(result.size() == 0) {
        			result.add(s);
        			result.add(p);
        			result.add(o);
        		} else {
        			for(int i = 0;i < result.sizeOrData ; i+=3) {
        				int existingP = result.data[i+1];
        				if(p == existingP) {
        					int existingO = result.data[i+2];
        					if(existingO == o) return;
        				}
        			}
        			result.add(s);
    				result.add(p);
    				result.add(o);
        		}
        		
        	}
        	
            @Override
            public void execute(ReadGraphImpl graph, int type) {
                AssertedStatements stms = AssertedStatements.queryEach(graph, type, r2, graph.processor, entry, null, NOPT);
                if(result == null) {
                	result = stms.getResult();
                } else {
                	if (first) {
                		IntArray ia = result;
                		result = new IntArray();
                		if(ia.data != null) {
                			for(int i = 0;i < ia.sizeOrData ; i+=3) addStatement(ia.data[i],ia.data[i+1],ia.data[i+2]);
                		}
                		first = false;
                	}
                	IntArray ia = stms.getResult();
            		if(ia.data != null) {
            			for(int i = 0;i < ia.sizeOrData ; i+=3) addStatement(ia.data[i],ia.data[i+1],ia.data[i+2]);
            		}
                }
            }

            @Override
            public void finished(ReadGraphImpl graph) {
            }

    		@Override
    		public void exception(ReadGraphImpl graph, Throwable throwable) {
    		}

        }
        
        AssertionMapProc amp = new AssertionMapProc();

        // This dependency could be cut
    	PrincipalTypes.queryEach(graph, r1, graph.processor, entry, null, amp);
    	
        return amp.result;
    	
    }
 	
    final static private void forSingleAssertion(ReadGraphImpl graph, final int r1, final int r2, final Statements entry, final TripleIntProcedure procedure) {
        
    	IntArray map = getAssertionMap(graph, r1, r2, entry);
    	if(map == null) {
            if(entry != null) entry.finish(graph, procedure);
        	else procedure.finished(graph);
            return;
    	}
        
        int size = map.size();
        if(size == 3) {

        	int s = map.data[0];
        	int p = map.data[1];
        	int o = map.data[2];
        	
            if(entry != null) {
            	entry.addOrSetFunctional(s,p,o);
            	entry.finish(graph, procedure);
            } else {
        		procedure.execute(graph, s,p,o);
        		procedure.finished(graph);
        	}

        } else if(size == 0) {

            if(entry != null) entry.finish(graph, procedure);
        	else procedure.finished(graph);
        	
        } else {

        	int candidateS = map.data[0];
        	int candidateP = map.data[1];
        	int candidateO = map.data[2];

			SuperTypes candidate = SuperTypes.queryEach(graph, candidateS, graph.processor, entry, null, NOP);
			if(candidate.isExcepted()) {
				if(entry != null) entry.except((Throwable)candidate.getResult());
				procedure.exception(graph, (Throwable)candidate.getResult());
				return;
			}
			IntSet candidateIs = candidate.getResult();
        	
			for(int i=3;i<map.size();i+=3) {

				int nextS = map.data[i];
				int nextP = map.data[i+1];
				int nextO = map.data[i+2];
        		
				if(nextS != candidateS) {

        			if(candidateIs.contains(nextS)) {
        				
        				// Next is a super type of candidate => ignore next
        				
        			} else {
        			
	        			SuperTypes next = SuperTypes.queryEach(graph, nextS, graph.processor, entry, null, NOP);
	        			if(next.isExcepted()) {
	        				if(entry != null) entry.except((Throwable)next.getResult());
	        				procedure.exception(graph, (Throwable)next.getResult());
	        				return;
	        			}
	        			IntSet nextIs = next.getResult();
	        			
	        			if(nextIs.contains(candidateS)) {

	        				// Candidate is a super type of next => next is the new candidate
	        				
	        				candidateS = nextS;
	        				candidateP = nextP;
	        				candidateO = nextO;
	        				candidateIs = nextIs;
	        				
	        			} else {

	        				// candidate and next are unrelated => error
	        				ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has conflicting assertions.", r1);
	        				
	        				if(entry != null) entry.except(exception);
	        				procedure.exception(graph, exception);
	        				return;	        				
	        				
	        			}
        			
        			}
        			
        		}
        		
        	}
        	
        	if(entry != null) {
            	entry.addOrSetFunctional(candidateS, candidateP, candidateO);
            	entry.finish(graph, procedure);
            } else {
        		procedure.execute(graph, candidateS, candidateP, candidateO);
        		procedure.finished(graph);
        	}
        	
        }
        
    }
    
    final static InternalProcedure<IntSet> NOP = new InternalProcedure<IntSet>() {

		@Override
		public void execute(ReadGraphImpl graph, IntSet result) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}
    	
    };
 	
 	// Search for one statement
 	final static public void computeFunctionalIndex(ReadGraphImpl graph, final int r1, final int r2, final Statements entry, final RelationInfo ri, final TripleIntProcedure procedure) {
        
        if(ri.isFinal) {
            
        	int result = graph.processor.querySupport.getFunctionalObject(r1, r2);

        	if(result == 0) {

            	// Check for assertions
            	forSingleAssertion(graph, r1, r2, entry, procedure);

        	} else if(result == -1) {

            	graph.processor.querySupport.getObjects(graph, r1, r2, new IntProcedure() {

            		@Override
            		public void execute(ReadGraphImpl graph, int i) {
            			if(entry != null) entry.addOrSetFunctional(r1, r2, i);
            			else procedure.execute(graph, r1, r2, i);
            		}

            		@Override
            		public void exception(ReadGraphImpl graph, Throwable t) {
            			if(DebugException.DEBUG) new DebugException(t).printStackTrace();
            		}

            		@Override
            		public void finished(ReadGraphImpl graph) {
            		}

            	});

            	// Check for assertions
            	forSingleAssertion(graph, r1, r2, entry, procedure);
        		
        	} else {

            	// If functional relation was found there is no need to check assertions
        		if(entry != null) {
        			entry.addOrSetFunctional(r1, r2, result);
        			entry.finish(graph, procedure);
        		} else {
        			procedure.execute(graph, r1, r2, result);
        			procedure.finished(graph);
        		}
        		
        	}

            
        } else {
            
            final AtomicBoolean found = new AtomicBoolean(false);
            
            // Note! The dependency is intentionally cut!
            DirectPredicates.queryEach(graph, r1, graph.processor, null, null, new SyncIntProcedure() {
                
                @Override
                public void run(ReadGraphImpl graph) {
                    
                    if(found.get()) {
                    	if(entry != null) entry.finish(graph, procedure);
                    	else procedure.finished(graph);
                    } else {
                    
	                    // Check for assertions
	                    forSingleAssertion(graph, r1, r2, entry, procedure);
	                    
                    }
                    
                }

                @Override
                public void execute(ReadGraphImpl graph, final int pred) {
                    
                    if(found.get()) return;

                    if(pred == r2) {
                        
                        inc();
                        
                        // Note! The dependency is intentionally cut!
                        DirectObjects.queryEach(graph, r1, pred, graph.processor, null, null, new IntProcedure() {

                            @Override
                            public void execute(ReadGraphImpl graph, int i) {
                                
                            	if(found.compareAndSet(false, true)) {
                                    
                                    if(entry != null) entry.addOrSetFunctional(r1, pred, i);
                                    else procedure.execute(graph, r1, pred, i);
                            		
                            	} else {

                 	            	ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has more than one statement.", r1);
                 	            	if(entry != null) entry.except(exception);
                            		procedure.exception(graph, exception);
                            		
                            	}

                            }

                            @Override
                            public void finished(ReadGraphImpl graph) {
                                dec(graph);
                            }
	        				
	        				@Override
	        				public void exception(ReadGraphImpl graph, Throwable t) {
	        					procedure.exception(graph, t);
                                dec(graph);
	        	            }

                        });

                    } else {
                        
                        inc();
                    
                        SuperRelations.queryEach(graph, pred, graph.processor, entry, null, new InternalProcedure<IntSet>() {
    
                            @Override
                            public void execute(ReadGraphImpl graph, IntSet result) {
                                
                                if(found.get()) {
                                    dec(graph);
                                    return;
                                }

                                if(result.contains(r2)) {
                                    
                                    inc();
                                    
                                    // Note! The dependency is intentionally cut!
                                    DirectObjects.queryEach(graph, r1, pred, graph.processor, null, null, new IntProcedure() {
    
                                        @Override
                                        public void execute(ReadGraphImpl graph, int i) {
                                            
                                        	if(found.compareAndSet(false, true)) {
                                                
                                            	if(entry != null) entry.addOrSetFunctional(r1, pred, i);
                                            	else procedure.execute(graph, r1, pred, i);
                                        		
                                        	} else {

                             	            	ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has more than one statement.", r1);
                             	            	if(entry != null) entry.except(exception);
                             	                procedure.exception(graph, exception);
                                        		
                                        	}
                                            
                                        }
    
                                        @Override
                                        public void finished(ReadGraphImpl graph) {
                                            dec(graph);
                                        }
            	        				
            	        				@Override
            	        				public void exception(ReadGraphImpl graph, Throwable t) {
            	        					procedure.exception(graph, t);
                                            dec(graph);
            	        	            }

                                    });
    
                                }
                                
                                dec(graph);
                                
                            }
	        				
	        				@Override
	        				public void exception(ReadGraphImpl graph, Throwable t) {
	        					procedure.exception(graph, t);
                                dec(graph);
	        	            }

                        });
                        
                    }
                    
                }

                @Override
                public void finished(ReadGraphImpl graph) {
                    
                    dec(graph);
                    
                }
                
            });
            
        }

 	}
    
    final static private void forAssertions(ReadGraphImpl graph, final int r1, final int r2, final Statements entry, final TripleIntProcedure procedure) {

        PrincipalTypes.queryEach(graph, r1, graph.processor, entry, null, new SyncIntProcedure() {
            
            @Override
            public void run(ReadGraphImpl graph) {
                
                if(entry != null) entry.finish(graph, procedure);
                else procedure.finished(graph);
                
            }
            
            TripleIntProcedure proc = new TripleIntProcedureAdapter() {

                @Override
                public void execute(ReadGraphImpl graph, int s, int p, int o) {
                	if(entry != null) entry.addOrSet(s, p, o);
                	else procedure.execute(graph, s, p, o);
                }

                @Override
                public void finished(ReadGraphImpl graph) {
                    dec(graph);
                }
				
				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
                    dec(graph);
					procedure.exception(graph, t);
	            }

            }; 

            @Override
            public void execute(ReadGraphImpl graph, int type) {

                inc();
                
                AssertedStatements.queryEach(graph, type, r2, graph.processor, entry, null, proc);
                
            }
            
            @Override
            public void finished(ReadGraphImpl graph) {
                dec(graph);       
            }
            
            @Override
            public void exception(ReadGraphImpl graph, Throwable t) {
                dec(graph);
            }
            
        });
        

    }

 	final static public void computeNotFunctionalIndex(ReadGraphImpl graph, final int r1, final int r2, final Statements entry,  final RelationInfo ri, final TripleIntProcedure procedure) {

 		if(ri.isFinal) {

 			graph.processor.querySupport.getObjects(graph, r1, r2, new IntProcedure() {

 				@Override
 				public void execute(ReadGraphImpl graph, int i) {
 					if(entry != null) entry.addOrSet(r1, r2, i);
 					else procedure.execute(graph, r1, r2, i);
 				}

 				@Override
 				public void exception(ReadGraphImpl graph, Throwable t) {
 					if(DebugException.DEBUG) new DebugException(t).printStackTrace();
 					procedure.exception(graph, t);
 				}

 				@Override
 				public void finished(ReadGraphImpl graph) {
 				}

 			});

 			if(ri.isAsserted) {
 	 			forAssertions(graph, r1, r2, entry, procedure);
 			} else {
 				if(entry != null) entry.finish(graph, procedure);
 				else procedure.finished(graph);
 			}

        } else {

            // Note! The dependency is intentionally cut!
            DirectPredicates.queryEach(graph, r1, graph.processor, null, null, new SyncIntProcedure() {
                
                @Override
                public void run(ReadGraphImpl graph) {
                    
                    forAssertions(graph, r1, r2, entry, procedure);
                    
                }

                @Override
                public void execute(ReadGraphImpl graph, final int pred2) {

                    if(pred2 == r2) {
                        
                        inc();
                        
                        // Note! The dependency is intentionally cut!
                        DirectObjects.queryEach(graph, r1, pred2, graph.processor, null, null, new IntProcedure() {

                            @Override
                            public void execute(ReadGraphImpl graph, int i) {
                            	if(entry != null) entry.addOrSet(r1, pred2, i);
                            	else procedure.execute(graph, r1, pred2, i);
                            }

                            @Override
                            public void finished(ReadGraphImpl graph) {
                                dec(graph);
                            }
	        				
	        				@Override
	        				public void exception(ReadGraphImpl graph, Throwable t) {
	        					procedure.exception(graph, t);
                                dec(graph);
	        	            }

                        });

                    } else {
                    
//                        inc();

                        try {
                            
                            IntSet result = SuperRelations.queryEach2(graph, pred2, graph.processor, entry, null, null);
                            if(result.contains(r2)) {

                                inc();

                                // Note! The dependency is intentionally cut!
                                DirectObjects.queryEach(graph, r1, pred2, graph.processor, null, null, new IntProcedure() {

                                    @Override
                                    public void execute(ReadGraphImpl graph, int i) {
                                        if(entry != null) entry.addOrSet(r1, pred2, i);
                                        else procedure.execute(graph, r1, pred2, i);

                                    }

                                    @Override
                                    public void finished(ReadGraphImpl graph) {
                                        dec(graph);
                                    }

                                    @Override
                                    public void exception(ReadGraphImpl graph, Throwable t) {
                                        procedure.exception(graph, t);
                                        dec(graph);
                                    }

                                });

                            }
                            
                        } catch (Throwable e) {
                            procedure.exception(graph, e);
                        }

                    }
                    
                }

                @Override
                public void finished(ReadGraphImpl graph) {
                    dec(graph);
                }
                
            });
            
        }
        
    }
    
    @Override
    public void computeForEach(ReadGraphImpl graph, final QueryProcessor provider, final TripleIntProcedure procedure, final boolean store) {
    	computeForEach(graph, r1(), r2(), this, procedure);
    }
 	
    public static void computeForEach(ReadGraphImpl graph, final int r1, final int r2, final Statements entry, final TripleIntProcedure procedure) {

    	RelationInfoQuery riEntry = RelationInfoQuery.probe(graph, r2);
    	if(riEntry != null) {
    		RelationInfo ri = riEntry.getResult();
        	graph.ensureLoaded(r1, r2);       
        	if(ri.isFunctional) {
        		computeFunctionalIndex(graph, r1, r2, entry, ri, procedure);
        	} else {
        		computeNotFunctionalIndex(graph, r1, r2, entry, ri, procedure);
        	}
    		return;
    	}
    	
        RelationInfoQuery.queryEach(graph, r2, graph.processor, entry, null, new InternalProcedure<RelationInfo>() {

            @Override
            public void execute(ReadGraphImpl graph, final RelationInfo ri) {
            	
            	graph.ensureLoaded(r1, r2);
            	if(ri.isFunctional) {
            		computeFunctionalIndex(graph, r1, r2, entry, ri, procedure);
            	} else {
            		computeNotFunctionalIndex(graph, r1, r2, entry, ri, procedure);
            	}
                
            }
			
			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				procedure.exception(graph, t);
            }

        });
        
    }
    
    @Override
    public String toString() {
    	return "Statements[" + r1() + " - " + r2() + "]";
    }

    final private void finish(ReadGraphImpl graph, TripleIntProcedure procedure) {
    	
        assert(assertPending());

//        ArrayList<TripleIntProcedure> p = null;

        synchronized(this) {

            setReady();
//            p = procs;
//            procs = null;

        }

        IntArray v = (IntArray)getResult();
        final IntArray value = (IntArray)getResult();

//        if(p != null) {
//
//	        for(TripleIntProcedure proc : p) {
//		        for(int i=0;i<value.size();i+=3) {
//		        	proc.execute(graph, value.data[i], value.data[i+1], value.data[i+2]);
//		        }
//	        }
//	        for(int i=0;i<value.size();i+=3) {
//	        	procedure.execute(graph, value.data[i], value.data[i+1], value.data[i+2]);
//	        }
//
//            for(TripleIntProcedure proc : p) proc.finished(graph);
//
//        }

        for(int i=0;i<value.size();i+=3) {
        	procedure.execute(graph, value.data[i], value.data[i+1], value.data[i+2]);
        }
        
        procedure.finished(graph);
    	
    }

//    final private void finish(ReadGraphImpl graph, QueryProcessor provider) {
//        
//    	assert(isPending());
//
//        ArrayList<TripleIntProcedure> p = null;
//
//        synchronized(this) {
//        	
//        	setReady();
//            p = procs;
//            procs = null; 
//        
//        }
//        
//        if(p != null) {
//        	
//	        final IntArray value = (IntArray)getResult();
//	        for(TripleIntProcedure proc : p) {
//		        for(int i=0;i<value.size();i+=3) {
//		        	proc.execute(graph, value.data[i], value.data[i+1], value.data[i+2]);
//		        }
//	        }
//
//	        for(TripleIntProcedure proc : p) proc.finished(graph);
//
//        }
//
//    }
    
    synchronized public void addOrSet(int s, int p, int o) {
        
    	assert(assertPending());
        
    	IntArray value = (IntArray)getResult();
        value.add(s);
        value.add(p);
        value.add(o);
        
    }

    final static public int r1(long id) {
        return (int)(id>>>32);
    }
    
    final static public int r2(long id) {
        return (int)id;
    }
    
    final public void addOrSetFunctional(int s, long po) {
    	
    	addOrSetFunctional(s, r1(po), r2(po));
    	
    }

    final public void addOrSetFunctional(int s, int p, int o) {
        
    	assert(assertPending());
        
    	IntArray value = (IntArray)getResult();
        value.add(s);
        value.add(p);
        value.add(o);
        
    }
    
    @Override
    public void performFromCache(ReadGraphImpl graph, QueryProcessor provider, final TripleIntProcedure procedure) {

    	assert(isReady());

    	if(handleException(graph, procedure)) return;
    	
        final IntArray value = (IntArray)getResult();
        for(int i=0;i<value.size();i+=3) {
        	procedure.execute(graph, value.data[i], value.data[i+1], value.data[i+2]);
        }

        procedure.finished(graph);
        
    }
    
    @Override
    public void recompute(ReadGraphImpl graph, QueryProcessor provider) {
        
        final Semaphore s = new Semaphore(0);
    	
        computeForEach(graph, provider, new TripleIntProcedureAdapter() {

            @Override
            public void finished(ReadGraphImpl graph) {
                s.release();
            }
			
			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				new Error("Error in recompute.", t).printStackTrace();
				s.release();
            }

        }, true);
        
    	while(!s.tryAcquire()) {
    		provider.resume(graph);
    	}
        
//        try {
//            s.acquire();
//        } catch (InterruptedException e) {
//            throw new Error(e);
//        }
        
    }

    @Override
    public int type() {
        return RequestFlags.IMMEDIATE_UPDATE;
    }
    

    @Override
    boolean isImmutable(ReadGraphImpl graph) {
    	return graph.processor.isImmutable(r1());
    }
    
}
