/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - gitlab #146 - tooltip support
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl.contribution;

import java.util.Collections;
import java.util.Map;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.simantics.Simantics;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.BuiltinKeys.LabelerKey;
import org.simantics.browsing.ui.DataSource;
import org.simantics.browsing.ui.GraphExplorer.ModificationContext;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryUpdater;
import org.simantics.browsing.ui.common.labelers.LabelerContent;
import org.simantics.browsing.ui.common.labelers.LabelerStub;
import org.simantics.browsing.ui.common.node.IModifiableNode;
import org.simantics.browsing.ui.graph.impl.contributor.labeler.ColumnLabelerContributorImpl;
import org.simantics.browsing.ui.graph.impl.request.ResourceQuery;
import org.simantics.db.ReadGraph;
import org.simantics.db.UndoContext;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.exception.InvalidVariableException;
import org.simantics.db.layer0.exception.PendingVariableException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.request.Read;
import org.simantics.utils.ui.ErrorLogger;

public abstract class FinalLabelerContributionImpl extends LabelerStub {

    protected final PrimitiveQueryUpdater       updater;
    private final ResourceQuery<LabelerContent> labelQuery;

    final protected NodeContext                context;
    final private BuiltinKeys.LabelerKey    key;

    public Object getIdentity(LabelerKey key) {
        return key;
    }

    public FinalLabelerContributionImpl(final PrimitiveQueryUpdater updater, final NodeContext context, final LabelerKey key) {

        this.updater = updater;
        this.context = context;
        this.key = key;

        labelQuery = new ResourceQuery<LabelerContent>(getIdentity(key), context) {

            @Override
            public LabelerContent perform(ReadGraph graph) throws DatabaseException {
                try {
                    int cat = category(graph, context);
                    Map<String, String> lbls = labels(graph, context);
                    // Make sure that null is not returned.
                    if (lbls == null)
                        throw new NullPointerException("FinalLabelerContributionImpl.labels is not allowed to return null, but " + FinalLabelerContributionImpl.this.getClass() + " just did it");
                    return new LabelerContent(cat, lbls);
                } catch (PendingVariableException e) {
                    return LabelerContent.NO_CONTENT;
                } catch (InvalidVariableException e) {
                    //ErrorLogger.defaultLogError("FinalLabelerContributionImpl.labelQuery failed due to invalid variable.", e);
                    return LabelerContent.NO_CONTENT;
                } catch (DatabaseException e) {
                    throw e;
                } catch (Throwable t) {
                    ErrorLogger.defaultLogError("FinalLabelerContributionImpl.labelQuery produced unexpected exception.", t);
                    return LabelerContent.NO_CONTENT;
                }
            }

            @Override
            public String toString() {
                return FinalLabelerContributionImpl.this + " with context " + context;
            }

        };

    }

    protected Procedure<LabelerContent> createProcedure() {

        return new Procedure<LabelerContent>() {

            @Override
            public void execute(LabelerContent result) {
                replaceResult(result);
            }

            @Override
            public void exception(Throwable t) {
                ErrorLogger.defaultLogError("FinalLabelerContributionImpl.labelQuery failed, see exception for details.", t);
            }

        };

    }

    protected void replaceResult(LabelerContent result) {
        setContent(result);
        updater.scheduleReplace(context, key, this);
    }

    @Override
    public Map<String, String> getLabels() {

        if (content == LabelerContent.NO_CONTENT) {

            final DataSource<ReadGraph> source = updater.getDataSource(ReadGraph.class);
            assert(source != null);

            final Procedure<LabelerContent> procedure = createProcedure();
            
            source.schedule(graph -> {
                if(procedure instanceof Listener<?>)
                    graph.asyncRequest(labelQuery, (Listener<LabelerContent>)procedure);
                else
                    graph.asyncRequest(labelQuery, procedure);
            });

        }

        if(content == null) return Collections.emptyMap();

        return content.labels;

    }

    @Override
    public Modifier getModifier(final ModificationContext modificationContext, final String columnKey) {
    	
        Object obj = context.getConstant(BuiltinKeys.INPUT);
        if (obj instanceof IModifiableNode) {
            return ((IModifiableNode) obj).getModifier(columnKey);
        }
        try {
			return Simantics.getSession().syncRequest(new Read<Modifier>() {

				@Override
				public Modifier perform(ReadGraph graph) throws DatabaseException {
				    UndoContext undoContext = null == modificationContext ? null :
				        (UndoContext)modificationContext.getAdapter(UndoContext.class);
					return FinalLabelerContributionImpl.this.getModifier(graph, undoContext, context, columnKey);
				}
				
			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
			return null;
		}
        
    }

    @Override
    public boolean shouldCreateToolTip(Event event, NodeContext nodeContext) {
        return createToolTip(event, nodeContext);
    }

    @Override
    public Composite createToolTipContentArea(Event event, Composite parent, NodeContext nodeContext) {
        return (Composite)createToolTipContent(event, parent, nodeContext);
    }

    // OVERRIDE

    public Modifier getModifier(ReadGraph graph, UndoContext undoContext, NodeContext context, String columnKey) throws DatabaseException {
    	return null;
    }

    public abstract Map<String, String> labels(ReadGraph graph, NodeContext context) throws DatabaseException;

    public abstract int category(ReadGraph graph, NodeContext context) throws DatabaseException;

    public boolean createToolTip(Object event, NodeContext nodeContext) {
        return false;
    }

    public Object createToolTipContent(Object event, Object parent, NodeContext nodeContext) {
        return null;
    }

}
