package org.simantics.db.layer0.variable;

import gnu.trove.map.hash.THashMap;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.simantics.databoard.binding.Binding;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.NoSingleResultException;

public class ConstantChildVariable extends AbstractChildVariable {

    final Variable parent;
    final String name;
    final Map<String, ConstantPropertyVariable> properties;

    public ConstantChildVariable(Variable parent, String name) {
        this.parent = parent;
        this.name = name;
        this.properties = new THashMap<String, ConstantPropertyVariable>();
    }

    public ConstantChildVariable(Variable parent, String name, Collection<ConstantPropertyVariableBuilder> propertyBuilders) {
    	this(parent, name);
    	for(ConstantPropertyVariableBuilder b : propertyBuilders) {
    		ConstantPropertyVariable property = b.build(this);
    		properties.put(b.getName(), property);
    	}
    }
    
    public ConstantChildVariable(Variable parent, String name, ConstantPropertyVariableBuilder ... propertyBuilders) {
    	this(parent, name);
    	for(ConstantPropertyVariableBuilder b : propertyBuilders) {
    		ConstantPropertyVariable property = b.build(this);
    		properties.put(b.getName(), property);
    	}
    }

    public ConstantChildVariable(Variable parent, String name, String[] propertyNames, Binding[] bindings, Object ... values) {

    	this(parent, name);

    	buildProperties(this, propertyNames, bindings, values, properties);

    }

    private Map<String, Variable> buildProperties(Variable parent, String[] propertyNames, Binding[] bindings, Object ... values) {

        assert parent != null;
        assert propertyNames.length == bindings.length;
        assert propertyNames.length == values.length;

    	Map<String, Variable> result = new HashMap<String, Variable>();
    	
    	for(int i=0;i<values.length;i++) {
    		String property = propertyNames[i];
    		result.put(property, new ConstantPropertyVariable(parent, property, values[i], bindings[i]));
    	}
    	
    	return result;

    }

    @Override
    public Map<String, Variable> collectDomainProperties(ReadGraph graph, Map<String, Variable> properties) throws DatabaseException {
    	if(properties == null) properties = new HashMap<String,Variable>(this.properties.size());
    	properties.putAll(this.properties);
    	return properties;
    }
    
    @Override
    public Variable getPossibleDomainProperty(ReadGraph graph, String name) throws DatabaseException {
    	return properties.get(name);
    }
    
    @Override
    public Variable getParent(ReadGraph graph) throws DatabaseException {
    	return parent;
    }
    
//    @Override
//    public Object getSerialized(ReadGraph graph) throws DatabaseException {
//    	return getName(graph);
//    }
    
    @Override
    public String getName(ReadGraph graph) throws DatabaseException {
    	return name;
    }

    @Override
    public String getLabel(ReadGraph graph) throws DatabaseException {
    	Variable variable = getPossibleDomainProperty(graph, Variables.LABEL);
    	if(variable != null) return variable.getValue(graph);
    	else return name;
    }

    @Override
    public Resource getRepresents(ReadGraph graph) throws DatabaseException {
    	return null;
//    	Variable variable = getPossibleDomainProperty(graph, Variables.REPRESENTS);
//    	if(variable != null) return variable.getValue(graph);
//    	else return null;
    }

    @Override
    public Resource getType(ReadGraph graph) throws DatabaseException {
    	return getDomainProperty(graph, Variables.TYPE).getValue(graph);
    }

    @Override
    public Resource getPossibleType(ReadGraph graph) throws DatabaseException {
    	Variable v = getPossibleDomainProperty(graph, Variables.TYPE);
    	return v != null ? v.<Resource>getPossibleValue(graph) : null;
    }

    @Override
    public Resource getType(ReadGraph graph, Resource baseType) throws DatabaseException {
    	Resource type = getPossibleType(graph, baseType);
    	if (type != null)
    		return type;
    	throw new NoSingleResultException("variable " + getPossibleURI(graph) + " has no type", -1);
    }

    @Override
    public Resource getPossibleType(ReadGraph graph, Resource baseType) throws DatabaseException {
    	Variable typev = getPossibleDomainProperty(graph, Variables.TYPE);
    	if (typev == null)
    		return null;
    	Resource type = typev.getValue(graph);
    	return type != null && graph.isInheritedFrom(type, baseType) ? type : null;
    }

    @Override
    public Collection<Variable> getProperties(ReadGraph graph, String classification) throws DatabaseException {
    	Collection<Variable> result = null;
    	Variable singleResult = null;
    	for(ConstantPropertyVariable property : properties.values()) {
    		if(property.getClassifications(graph).contains(classification)) {
    			if(result == null && singleResult == null) singleResult = property;
    			else if(result == null && singleResult != null) {
    				result = new ArrayList<Variable>();
    				result.add(singleResult);
    				result.add(property);
    			} else {
    				result.add(property);
    			}
    		}
    	}
    	if(result != null) return result;
    	if(singleResult != null) return Collections.singletonList(singleResult);
    	else return Collections.emptyList();
    } 

}
